from pathlib import Path

import pytest

from debputy.util import compute_output_filename


def write_unpacked_deb(
    root: Path,
    package: str,
    version: str,
    arch: str,
    arch_variant: str | None,
) -> None:
    lines = [
        f"Package: {package}",
        f"Version: {version}",
        f"Architecture: {arch}",
    ]
    if arch_variant:
        lines.append(f"Architecture-Variant: {arch_variant}")
    (root / "control").write_text("".join(f"{x}\n" for x in lines))


@pytest.mark.parametrize(
    "package,version,arch,is_udeb,arch_variant,expected",
    [
        ("fake", "1.0", "amd64", False, None, "fake_1.0_amd64.deb"),
        ("fake", "1.0", "amd64", True, None, "fake_1.0_amd64.udeb"),
        ("fake", "2:1.0", "amd64", False, None, "fake_1.0_amd64.deb"),
        ("fake", "2:1.0", "amd64", True, None, "fake_1.0_amd64.udeb"),
        ("fake", "3.0", "all", False, None, "fake_3.0_all.deb"),
        ("fake", "3.0", "all", True, None, "fake_3.0_all.udeb"),
        # Architecture-Variant
        ("fake", "1.0", "amd64", False, "i386", "fake_1.0_i386.deb"),
        ("fake", "1.0", "amd64", True, "i386", "fake_1.0_i386.udeb"),
    ],
)
def test_generate_deb_filename(
    tmp_path: Path,
    package: str,
    version: str,
    arch: str,
    is_udeb: bool,
    arch_variant: str | None,
    expected: str,
) -> None:
    write_unpacked_deb(tmp_path, package, version, arch, arch_variant)
    assert compute_output_filename(str(tmp_path), is_udeb) == expected
