/*
  This file is part of TALER
  Copyright (C) 2025 Taler Systems SA

  TALER is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published
  by the Free Software Foundation; either version 3, or (at your
  option) any later version.

  TALER is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public
  License along with TALER; see the file COPYING.  If not, see
  <http://www.gnu.org/licenses/>
*/

/**
 * @file lib/donau_api_charity_patch.c
 * @brief Implementation of the PATCH /charities/$ID call for the donau HTTP API
 * @author Bohdan Potuzhnyi
 */
#include <jansson.h>
#include <microhttpd.h>
#include <gnunet/gnunet_curl_lib.h>
#include <taler/taler_curl_lib.h>
#include <taler/taler_json_lib.h>
#include "donau_service.h"
#include "donau_api_curl_defaults.h"
#include "donau_json_lib.h"


/**
 * Handle for a PATCH /charities/$ID request.
 */
struct DONAU_CharityPatchHandle
{
  /**
   * Fully qualified request URL.
   */
  char *url;

  /**
   * CURL job for the request.
   */
  struct GNUNET_CURL_Job *job;

  /**
   * Callback for the response.
   */
  DONAU_PatchCharityResponseCallback cb;

  /**
   * Closure for @e cb.
   */
  void *cb_cls;

  /**
   * Reference to the CURL context.
   */
  struct GNUNET_CURL_Context *ctx;

  /**
   * Helper context for POST-style uploads.
   */
  struct TALER_CURL_PostContext post_ctx;
};


/**
 * Finalizer called once the PATCH request is complete.
 */
static void
handle_charity_patch_finished (void *cls,
                               long response_code,
                               const void *resp_obj)
{
  struct DONAU_CharityPatchHandle *cph = cls;
  const json_t *j = resp_obj;
  struct DONAU_PatchCharityResponse pcresp = {
    .hr.reply = j,
    .hr.http_status = (unsigned int) response_code
  };

  cph->job = NULL;
  switch (response_code)
  {
  case 0:
    pcresp.hr.ec = TALER_EC_GENERIC_INVALID_RESPONSE;
    GNUNET_break (0);
    break;
  case MHD_HTTP_OK:
  case MHD_HTTP_NO_CONTENT:
    /* nothing further to parse */
    break;
  case MHD_HTTP_BAD_REQUEST:
  case MHD_HTTP_NOT_FOUND:
  case MHD_HTTP_FORBIDDEN:
  case MHD_HTTP_UNAUTHORIZED:
  case MHD_HTTP_CONFLICT:
  case MHD_HTTP_INTERNAL_SERVER_ERROR:
    pcresp.hr.ec = TALER_JSON_get_error_code (j);
    pcresp.hr.hint = TALER_JSON_get_error_hint (j);
    break;
  default:
    pcresp.hr.ec = TALER_JSON_get_error_code (j);
    pcresp.hr.hint = TALER_JSON_get_error_hint (j);
    GNUNET_break_op (0);
    GNUNET_log (GNUNET_ERROR_TYPE_ERROR,
                "Unexpected response code %ld for PATCH %s\n",
                response_code,
                cph->url);
    break;
  }

  if (NULL != cph->cb)
  {
    cph->cb (cph->cb_cls,
             &pcresp);
    cph->cb = NULL;
  }
  DONAU_charity_patch_cancel (cph);
}


struct DONAU_CharityPatchHandle *
DONAU_charity_patch (
  struct GNUNET_CURL_Context *ctx,
  const char *url,
  const uint64_t charity_id,
  const char *charity_name,
  const char *charity_url,
  const struct TALER_Amount *max_per_year,
  const struct DONAU_CharityPublicKeyP *charity_pub,
  const struct DONAU_BearerToken *bearer,
  DONAU_PatchCharityResponseCallback cb,
  void *cb_cls)
{
  struct DONAU_CharityPatchHandle *cph;
  CURL *eh;
  json_t *body;

  if ( (NULL == charity_name) ||
       (NULL == charity_url) ||
       (NULL == max_per_year) ||
       (NULL == charity_pub) )
  {
    /* Caller must provide the complete CharityRequest payload. */
    GNUNET_break_op (0);
    return NULL;
  }

  body = GNUNET_JSON_PACK (
    GNUNET_JSON_pack_data_auto ("charity_pub",
                                charity_pub),
    GNUNET_JSON_pack_string ("charity_url",
                             charity_url),
    GNUNET_JSON_pack_string ("charity_name",
                             charity_name),
    TALER_JSON_pack_amount ("max_per_year",
                            max_per_year));
  if (NULL == body)
  {
    GNUNET_break (0);
    return NULL;
  }

  cph = GNUNET_new (struct DONAU_CharityPatchHandle);
  cph->ctx = ctx;
  cph->cb = cb;
  cph->cb_cls = cb_cls;

  {
    char *path;

    GNUNET_asprintf (&path,
                     "charities/%llu",
                     (unsigned long long) charity_id);
    cph->url = TALER_url_join (url,
                               path,
                               NULL);
    GNUNET_free (path);
  }
  if (NULL == cph->url)
  {
    json_decref (body);
    GNUNET_free (cph);
    GNUNET_break (0);
    return NULL;
  }

  eh = DONAU_curl_easy_get_ (cph->url);
  if (NULL == eh)
  {
    json_decref (body);
    GNUNET_free (cph->url);
    GNUNET_free (cph);
    GNUNET_break (0);
    return NULL;
  }
  if (GNUNET_OK !=
      TALER_curl_easy_post (&cph->post_ctx,
                            eh,
                            body))
  {
    GNUNET_break (0);
    curl_easy_cleanup (eh);
    json_decref (body);
    GNUNET_free (cph->url);
    GNUNET_free (cph);
    return NULL;
  }
  json_decref (body);
  GNUNET_assert (CURLE_OK ==
                 curl_easy_setopt (eh,
                                   CURLOPT_CUSTOMREQUEST,
                                   MHD_HTTP_METHOD_PATCH));
  cph->job = GNUNET_CURL_job_add2 (ctx,
                                   eh,
                                   cph->post_ctx.headers,
                                   &handle_charity_patch_finished,
                                   cph);
  if (NULL == cph->job)
  {
    GNUNET_break (0);
    TALER_curl_easy_post_finished (&cph->post_ctx);
    curl_easy_cleanup (eh);
    GNUNET_free (cph->url);
    GNUNET_free (cph);
    return NULL;
  }

  if (NULL != bearer)
  {
    struct curl_slist *auth;
    char *hdr;

    GNUNET_asprintf (&hdr,
                     "%s: Bearer %s",
                     MHD_HTTP_HEADER_AUTHORIZATION,
                     bearer->token);
    auth = curl_slist_append (NULL,
                              hdr);
    GNUNET_free (hdr);
    GNUNET_CURL_extend_headers (cph->job,
                                auth);
    curl_slist_free_all (auth);
  }

  return cph;
}


void
DONAU_charity_patch_cancel (
  struct DONAU_CharityPatchHandle *cph)
{
  if (NULL == cph)
    return;
  if (NULL != cph->job)
  {
    GNUNET_CURL_job_cancel (cph->job);
    cph->job = NULL;
  }
  TALER_curl_easy_post_finished (&cph->post_ctx);
  GNUNET_free (cph->url);
  GNUNET_free (cph);
}


/* end of donau_api_charity_patch.c */
