/*
  This file is part of TALER
  Copyright (C) 2020-2024 Taler Systems SA

  TALER is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as
  published by the Free Software Foundation; either version 3, or
  (at your option) any later version.

  TALER is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public
  License along with TALER; see the file COPYING.  If not, see
  <http://www.gnu.org/licenses/>
*/
/**
 * @file testing_api_cmd_post_donau_charity_merchant.c
 * @brief command to test POST /donau charity with merchant_pub
 * @author Bohdan Potuzhnyi
 * @author Vlada Svirsh
 */
#include "platform.h"
#include <taler/taler_exchange_service.h>
#include <taler/taler_testing_lib.h>
#include <taler/taler_signatures.h>
#include "taler_merchant_service.h"
#include "taler_merchant_testing_lib.h"
#include "taler_merchant_donau.h"
#include <donau/donau_service.h>
#include <donau/donau_testing_lib.h>


/**
 * State for a "status" CMD.
 */
struct StatusState
{
  /**
   * Handle to the "charity status" operation.
   */
  struct DONAU_CharityPostHandle *cph;

  /**
   * name of the charity
   */
  const char *charity_name;

  /**
   * charity url
   */
  const char *charity_url;

  /**
   * public key of the charity
   */
  struct DONAU_CharityPublicKeyP charity_pub;

  /**
   * Max donation amount for this charitiy and @e current_year.
   */
  struct TALER_Amount max_per_year;

  /**
   * The bearer token for authorization.
   */
  const struct DONAU_BearerToken *bearer;

  /**
   * Expected HTTP response code.
   */
  unsigned int expected_response_code;

  /**
   * Interpreter state.
   */
  struct TALER_TESTING_Interpreter *is;

  /**
   * charity id
   */
  uint64_t charity_id;

  /**
   * Merchant reference to fetch public key.
   */
  const char *merchant_reference;
};

/**
 * Offer internal data from a CMD, to other commands.
 *
 * @param cls closure.
 * @param[out] ret result.
 * @param trait name of the trait.
 * @param index index number of the object to offer.
 * @return #GNUNET_OK on success.
 */
static enum GNUNET_GenericReturnValue
charity_post_traits (void *cls,
                     const void **ret,
                     const char *trait,
                     unsigned int index)
{
  struct StatusState *ss = cls;
  struct TALER_TESTING_Trait traits[] = {
    TALER_TESTING_make_trait_charity_pub (&ss->charity_pub),
    TALER_TESTING_make_trait_charity_id (&ss->charity_id),
    TALER_TESTING_trait_end ()
  };

  return TALER_TESTING_get_trait (traits,
                                  ret,
                                  trait,
                                  index);
}


/**
 * Check that the reserve balance and HTTP response code are
 * both acceptable.
 *
 * @param cls closure.
 * @param gcr HTTP response details
 */
static void
charity_status_cb (void *cls,
                   const struct DONAU_PostCharityResponse *gcr)
{
  struct StatusState *ss = cls;

  ss->cph = NULL;
  if (ss->expected_response_code != gcr->hr.http_status)
  {
    GNUNET_log (GNUNET_ERROR_TYPE_ERROR,
                "Unexpected HTTP response code: %d in %s:%u\n",
                gcr->hr.http_status,
                __FILE__,
                __LINE__);
    json_dumpf (gcr->hr.reply,
                stderr,
                0);
    TALER_TESTING_interpreter_fail (ss->is);
    return;
  }
  if (ss->expected_response_code == gcr->hr.http_status)
    ss->charity_id = (unsigned long long) gcr->details.ok.charity_id;
  TALER_TESTING_interpreter_next (ss->is);
}


/**
 * Run the command.
 *
 * @param cls closure.
 * @param cmd the command being executed.
 * @param is the interpreter state.
 */
static void
charity_status_run (void *cls,
                    const struct TALER_TESTING_Command *cmd,
                    struct TALER_TESTING_Interpreter *is)
{
  struct StatusState *ss = cls;

  ss->is = is;

  if (NULL != ss->merchant_reference)
  {
    const struct TALER_TESTING_Command *mc;
    const struct TALER_MerchantPublicKeyP *mpub;

    mc = TALER_TESTING_interpreter_lookup_command (is,
                                                   ss->merchant_reference);
    GNUNET_assert (NULL != mc);
    GNUNET_assert (GNUNET_OK ==
                   TALER_TESTING_get_trait_merchant_pub (mc,
                                                         &mpub));

    ss->charity_pub.eddsa_pub = mpub->eddsa_pub;
  }

  ss->cph = DONAU_charity_post (
    TALER_TESTING_interpreter_get_context (is),
    TALER_TESTING_get_donau_url (is),
    ss->charity_name,
    ss->charity_url,
    &ss->max_per_year,
    &ss->charity_pub,
    ss->bearer,
    &charity_status_cb,
    ss);
}


/**
 * Cleanup the state from a "reserve status" CMD, and possibly
 * cancel a pending operation thereof.
 *
 * @param cls closure.
 * @param cmd the command which is being cleaned up.
 */
static void
cleanup (void *cls,
         const struct TALER_TESTING_Command *cmd)
{
  struct StatusState *ss = cls;

  if (NULL != ss->cph)
  {
    // log incomplete command
    TALER_TESTING_command_incomplete (ss->is,
                                      cmd->label);
    DONAU_charity_post_cancel (ss->cph);
    ss->cph = NULL;
  }
  GNUNET_free (ss);
}


struct TALER_TESTING_Command
TALER_TESTING_cmd_charity_post_merchant (
  const char *label,
  const char *name,
  const char *url,
  const char *max_per_year,
  const struct DONAU_BearerToken *bearer,
  const char *merchant_reference,
  unsigned int expected_response_code)
{
  struct StatusState *ss;

  ss = GNUNET_new (struct StatusState);
  ss->merchant_reference = merchant_reference;
  ss->charity_name = name;
  ss->charity_url = url;
  // parse string max_per_year to amount
  if (GNUNET_OK !=
      TALER_string_to_amount (max_per_year,
                              &ss->max_per_year))
  {
    GNUNET_log (GNUNET_ERROR_TYPE_ERROR,
                "Failed to parse amount `%s' at %s\n",
                max_per_year,
                label);
    GNUNET_assert (0);
  }
  ss->expected_response_code = expected_response_code;
  ss->bearer = bearer;
  {
    struct TALER_TESTING_Command cmd = {
      .cls = ss,
      .label = label,
      .run = &charity_status_run,
      .cleanup = &cleanup,
      .traits = &charity_post_traits
    };

    return cmd;
  }
}
