/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE support@mankowski.fr  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
/** @file
 * A report class for document
 *
 * @author Stephane MANKOWSKI
 */
#include "skgreportbank.h"
#include "skgtraces.h"
#include "skgdocumentbank.h"
#include "skgaccountobject.h"
#include "skgunitobject.h"

#include <kstandarddirs.h>
#include <grantlee/engine.h>
#include <grantlee/templateloader.h>
#ifdef SKGGRANTLEE17ENABLED
#include <grantlee/qtlocalizer.h>
#include <grantlee/metatype.h>
#endif
#include <kaboutdata.h>
#include <kcomponentdata.h>
#include <kcolorscheme.h>
#include <KUrl>
#include <KGlobalSettings>

#include <QTextStream>
#include <QFile>
#include <QDir>
#include <QFont>


#ifdef SKGGRANTLEE17ENABLED
GRANTLEE_BEGIN_LOOKUP(SKGObjectBase)
Q_UNUSED(object);
Q_UNUSED(property);
GRANTLEE_END_LOOKUP
#endif

SKGReport::SKGReport(SKGDocument* iDocument)
    : QObject(), m_document(iDocument)
{
    SKGTRACEIN(1, "SKGReport::SKGReport");

    //Grantlee initialization
#ifdef SKGGRANTLEE17ENABLED
    Grantlee::MetaType::init();
    Grantlee::registerMetaType<SKGObjectBase>();
#endif
}

SKGReport::~SKGReport()
{
    SKGTRACEIN(1, "SKGReport::~SKGReport");
}

void SKGReport::setMonth(const QString& iMonth)
{
    m_cache["getMonth"] = iMonth;
}

QString SKGReport::getMonth()
{
    QString month = m_cache["getMonth"].toString();
    if (month.isEmpty()) {
        month = QDate::currentDate().toString("yyyy-MM");
        m_cache["getMonth"] = month;
    }
    return month;
}

QString SKGReport::getPreviousMonth()
{
    QString previousmonth = m_cache["getPreviousMonth"].toString();
    if (previousmonth.isEmpty()) {
        QString month = getMonth();
        if (!month.isEmpty()) {
            QDate date = QDate::fromString(month, "yyyy-MM");
            QDate date1 = date.addDays(-1);
            previousmonth = date1.toString("yyyy-MM");
        }
        m_cache["getPreviousMonth"] = previousmonth;
    }
    return previousmonth;
}

void SKGReport::addItemsInMapping(QVariantHash& iMapping)
{
    iMapping.insert("report", QVariant::fromValue<QObject*>(this));
    iMapping.insert("current_date", QDate::currentDate());
    KColorScheme scheme(QPalette::Normal, KColorScheme::Window);
    iMapping.insert("color_negativetext", scheme.foreground(KColorScheme::NegativeText).color().name().right(6));
    iMapping.insert("color_positivetext", scheme.foreground(KColorScheme::PositiveText).color().name().right(6));
    iMapping.insert("color_neutraltext", scheme.foreground(KColorScheme::NeutralText).color().name().right(6));
    iMapping.insert("color_normaltext", scheme.foreground(KColorScheme::NormalText).color().name().right(6));
    iMapping.insert("color_inactivetext", scheme.foreground(KColorScheme::InactiveText).color().name().right(6));
    iMapping.insert("color_activetext", scheme.foreground(KColorScheme::ActiveText).color().name().right(6));
    iMapping.insert("color_linktext", scheme.foreground(KColorScheme::LinkText).color().name().right(6));
    iMapping.insert("color_visitedtext", scheme.foreground(KColorScheme::VisitedText).color().name().right(6));
    iMapping.insert("color_normalbackground", scheme.background(KColorScheme::NormalBackground).color().name().right(6));
    iMapping.insert("color_activebackground", scheme.background(KColorScheme::ActiveBackground).color().name().right(6));

    QFont generalFont = KGlobalSettings::generalFont();
    iMapping.insert("font_family", generalFont.family());
}

SKGError SKGReport::getReportFromTemplate(SKGReport* iReport, const QString& iFile, QString& oHtml)
{
    SKGError err;
    SKGTRACEINRC(10, "SKGReport::getReportFromTemplate", err);
    SKGDocument* doc = (iReport ? static_cast<SKGDocument*>(iReport->m_document) : NULL);

    //Prepare grantlee engine
    Grantlee::Engine gEngine;
    const QStringList pluginDirs = KGlobal::dirs()->resourceDirs("qtplugins");
    foreach(const QString & pluginDir, pluginDirs) {
        gEngine.addPluginPath(pluginDir);
    }
    gEngine.addDefaultLibrary("grantlee_skgfilters");

    Grantlee::FileSystemTemplateLoader::Ptr gLoader = Grantlee::FileSystemTemplateLoader::Ptr(new Grantlee::FileSystemTemplateLoader());
    gLoader->setTemplateDirs(QStringList(QFileInfo(iFile).dir().absolutePath()));
    gEngine.addTemplateLoader(gLoader);

    Grantlee::Template gTemplate = gEngine.loadByName(QFileInfo(iFile).fileName());

    QVariantHash mapping;
    if (iReport) iReport->addItemsInMapping(mapping);
    if (doc) mapping.insert("document", QVariant::fromValue<QObject*>(doc));

    QString dir = "file://" % QFileInfo(KStandardDirs::locate("data", "kdeui/about/kde_infopage.css")).dir().absolutePath() % '/';
    {
        QFile file(KStandardDirs::locate("data", "kdeui/about/kde_infopage.css"));
        if (file.open(QIODevice::ReadOnly | QIODevice::Text))
            mapping.insert("kde_infopage_css", QString(file.readAll()).replace("url(", "url(" % dir));
    }

    KAboutData* about = const_cast<KAboutData*>(KGlobal::mainComponent().aboutData());
    if (about) {
        mapping.insert("about_welcome", i18nc("Welcome message", "Welcome to %1", about->programName()));
        mapping.insert("about_programname", about->programName());
        mapping.insert("about_version", about->version());
        mapping.insert("about_bugaddress", about->bugAddress());
        mapping.insert("about_copyrightstatement", about->copyrightStatement());
        mapping.insert("about_homepage", about->homepage());
        mapping.insert("about_license", about->license());
        mapping.insert("about_shortdescription", about->shortDescription());
        mapping.insert("about_othertext", about->otherText());
    }
    Grantlee::Context gContext(mapping);

    //Generation
    {
        SKGTRACEINRC(10, "SKGReport::getReportFromTemplate-grantlee render", err);
        oHtml = gTemplate->render(&gContext);
        if (gTemplate->error()) err = SKGError(gTemplate->error(), gTemplate->errorString());
    }
    return err;
}

#include "skgreport.moc"
