﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/shield/ShieldRequest.h>
#include <aws/shield/Shield_EXPORTS.h>
#include <aws/shield/model/TimeRange.h>

#include <utility>

namespace Aws {
namespace Shield {
namespace Model {

/**
 */
class ListAttacksRequest : public ShieldRequest {
 public:
  AWS_SHIELD_API ListAttacksRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ListAttacks"; }

  AWS_SHIELD_API Aws::String SerializePayload() const override;

  AWS_SHIELD_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The ARNs (Amazon Resource Names) of the resources that were attacked. If you
   * leave this blank, all applicable resources for this account will be
   * included.</p>
   */
  inline const Aws::Vector<Aws::String>& GetResourceArns() const { return m_resourceArns; }
  inline bool ResourceArnsHasBeenSet() const { return m_resourceArnsHasBeenSet; }
  template <typename ResourceArnsT = Aws::Vector<Aws::String>>
  void SetResourceArns(ResourceArnsT&& value) {
    m_resourceArnsHasBeenSet = true;
    m_resourceArns = std::forward<ResourceArnsT>(value);
  }
  template <typename ResourceArnsT = Aws::Vector<Aws::String>>
  ListAttacksRequest& WithResourceArns(ResourceArnsT&& value) {
    SetResourceArns(std::forward<ResourceArnsT>(value));
    return *this;
  }
  template <typename ResourceArnsT = Aws::String>
  ListAttacksRequest& AddResourceArns(ResourceArnsT&& value) {
    m_resourceArnsHasBeenSet = true;
    m_resourceArns.emplace_back(std::forward<ResourceArnsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The start of the time period for the attacks. This is a
   * <code>timestamp</code> type. The request syntax listing for this call indicates
   * a <code>number</code> type, but you can provide the time in any valid <a
   * href="https://docs.aws.amazon.com/cli/latest/userguide/cli-usage-parameters-types.html#parameter-type-timestamp">timestamp
   * format</a> setting. </p>
   */
  inline const TimeRange& GetStartTime() const { return m_startTime; }
  inline bool StartTimeHasBeenSet() const { return m_startTimeHasBeenSet; }
  template <typename StartTimeT = TimeRange>
  void SetStartTime(StartTimeT&& value) {
    m_startTimeHasBeenSet = true;
    m_startTime = std::forward<StartTimeT>(value);
  }
  template <typename StartTimeT = TimeRange>
  ListAttacksRequest& WithStartTime(StartTimeT&& value) {
    SetStartTime(std::forward<StartTimeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The end of the time period for the attacks. This is a <code>timestamp</code>
   * type. The request syntax listing for this call indicates a <code>number</code>
   * type, but you can provide the time in any valid <a
   * href="https://docs.aws.amazon.com/cli/latest/userguide/cli-usage-parameters-types.html#parameter-type-timestamp">timestamp
   * format</a> setting. </p>
   */
  inline const TimeRange& GetEndTime() const { return m_endTime; }
  inline bool EndTimeHasBeenSet() const { return m_endTimeHasBeenSet; }
  template <typename EndTimeT = TimeRange>
  void SetEndTime(EndTimeT&& value) {
    m_endTimeHasBeenSet = true;
    m_endTime = std::forward<EndTimeT>(value);
  }
  template <typename EndTimeT = TimeRange>
  ListAttacksRequest& WithEndTime(EndTimeT&& value) {
    SetEndTime(std::forward<EndTimeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>When you request a list of objects from Shield Advanced, if the response does
   * not include all of the remaining available objects, Shield Advanced includes a
   * <code>NextToken</code> value in the response. You can retrieve the next batch of
   * objects by requesting the list again and providing the token that was returned
   * by the prior call in your request. </p> <p>You can indicate the maximum number
   * of objects that you want Shield Advanced to return for a single call with the
   * <code>MaxResults</code> setting. Shield Advanced will not return more than
   * <code>MaxResults</code> objects, but may return fewer, even if more objects are
   * still available.</p> <p>Whenever more objects remain that Shield Advanced has
   * not yet returned to you, the response will include a <code>NextToken</code>
   * value.</p> <p>On your first call to a list operation, leave this setting
   * empty.</p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  ListAttacksRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The greatest number of objects that you want Shield Advanced to return to the
   * list request. Shield Advanced might return fewer objects than you indicate in
   * this setting, even if more objects are available. If there are more objects
   * remaining, Shield Advanced will always also return a <code>NextToken</code>
   * value in the response.</p> <p>The default setting is 20.</p>
   */
  inline int GetMaxResults() const { return m_maxResults; }
  inline bool MaxResultsHasBeenSet() const { return m_maxResultsHasBeenSet; }
  inline void SetMaxResults(int value) {
    m_maxResultsHasBeenSet = true;
    m_maxResults = value;
  }
  inline ListAttacksRequest& WithMaxResults(int value) {
    SetMaxResults(value);
    return *this;
  }
  ///@}
 private:
  Aws::Vector<Aws::String> m_resourceArns;

  TimeRange m_startTime;

  TimeRange m_endTime;

  Aws::String m_nextToken;

  int m_maxResults{0};
  bool m_resourceArnsHasBeenSet = false;
  bool m_startTimeHasBeenSet = false;
  bool m_endTimeHasBeenSet = false;
  bool m_nextTokenHasBeenSet = false;
  bool m_maxResultsHasBeenSet = false;
};

}  // namespace Model
}  // namespace Shield
}  // namespace Aws
