//------------------------------------------------------------------------------
// <copyright file="FileVersionInfo.cs" company="Microsoft">
//     Copyright (c) Microsoft Corporation.  All rights reserved.
// </copyright>                                                                
//------------------------------------------------------------------------------

namespace System.Diagnostics {
    using Microsoft.Win32;
    using System.Runtime.Serialization.Formatters;
    using System.Text;
    using System.Configuration.Assemblies;
    using System.Runtime.InteropServices;
    using System.IO;
    using System.Security;
    using System.Security.Permissions;
    using System;
    using System.Globalization;
    using System.Runtime.Versioning;

    /// <devdoc>
    ///    <para>Provides version information for a physical file on disk.</para>
    /// </devdoc>
    [
    // Disabling partial trust scenarios
    PermissionSet(SecurityAction.LinkDemand, Name="FullTrust")
    ]
    public sealed class FileVersionInfo {

        private string fileName;
        private string companyName;
        private string fileDescription;
        private string fileVersion;
        private string internalName;
        private string legalCopyright;
        private string originalFilename;
        private string productName;
        private string productVersion;
        private string comments;
        private string legalTrademarks;
        private string privateBuild;
        private string specialBuild;
        private string language;
        private int fileMajor;
        private int fileMinor;
        private int fileBuild;
        private int filePrivate;
        private int productMajor;
        private int productMinor;
        private int productBuild;
        private int productPrivate;
        private int fileFlags;

        private FileVersionInfo(string fileName) {
            this.fileName = fileName;
        }

        /// <devdoc>
        ///    <para>Gets the comments associated with the file.</para>
        /// </devdoc>
        public string Comments {
            get {
                return comments;
            }
        }

        /// <devdoc>
        ///    <para>Gets the name of the company that produced the file.</para>
        /// </devdoc>
        public string CompanyName {
            get {
                return companyName;
            }
        }

        /// <devdoc>
        ///    <para>Gets the build number of the file.</para>
        /// </devdoc>
        public int FileBuildPart {
            get {
                return fileBuild;
            }
        }

        /// <devdoc>
        ///    <para>Gets the description of the file.</para>
        /// </devdoc>
        public string FileDescription {
            get {
                return fileDescription;
            }
        }

        /// <devdoc>
        ///    <para>Gets the major part of the version number.</para>
        /// </devdoc>
        public int FileMajorPart {
            get {
                return fileMajor;
            }
        }

        /// <devdoc>
        ///    <para>Gets the minor
        ///       part of the version number of the file.</para>
        /// </devdoc>
        public int FileMinorPart {
            get {
                return fileMinor;
            }
        }

        /// <devdoc>
        ///    <para>Gets the name of the file that this instance of System.Windows.Forms.FileVersionInfo
        ///       describes.</para>
        /// </devdoc>
        public string FileName {
            get {
                new FileIOPermission(FileIOPermissionAccess.PathDiscovery, fileName).Demand();
                return fileName;
            }
        }

        /// <devdoc>
        ///    <para>Gets the file private part number.</para>
        /// </devdoc>
        public int FilePrivatePart {
            get {
                return filePrivate;
            }
        }

        /// <devdoc>
        ///    <para>Gets the file version number.</para>
        /// </devdoc>
        public string FileVersion {
            get {
                return fileVersion;
            }
        }

        /// <devdoc>
        ///    <para>Gets the internal name of the file, if one exists.</para>
        /// </devdoc>
        public string InternalName {
            get {
                return internalName;
            }
        }

        /// <devdoc>
        ///    <para>Gets a value that specifies whether the file
        ///       contains debugging information or is compiled with debugging features enabled.</para>
        /// </devdoc>
        public bool IsDebug {
            get {
                return (fileFlags & NativeMethods.VS_FF_DEBUG) != 0;
            }
        }

        /// <devdoc>
        ///    <para>Gets a value that specifies whether the file has been modified and is not identical to
        ///       the original shipping file of the same version number.</para>
        /// </devdoc>
        public bool IsPatched {
            get {
                return (fileFlags & NativeMethods.VS_FF_PATCHED) != 0;
            }
        }

        /// <devdoc>
        ///    <para>Gets a value that specifies whether the file was built using standard release procedures.</para>
        /// </devdoc>
        public bool IsPrivateBuild {
            get {
                return (fileFlags & NativeMethods.VS_FF_PRIVATEBUILD) != 0;
            }
        }

        /// <devdoc>
        ///    <para>Gets a value that specifies whether the file
        ///       is a development version, rather than a commercially released product.</para>
        /// </devdoc>
        public bool IsPreRelease {
            get {
                return (fileFlags & NativeMethods.VS_FF_PRERELEASE) != 0;
            }
        }

        /// <devdoc>
        ///    <para>Gets a value that specifies whether the file is a special build.</para>
        /// </devdoc>
        public bool IsSpecialBuild {
            get {
                return (fileFlags & NativeMethods.VS_FF_SPECIALBUILD) != 0;
            }
        }

        /// <devdoc>
        ///    <para>
        ///       Gets the default language string for the version info block.
        ///    </para>
        /// </devdoc>
        public string Language {
            get {
                return language;
            }
        }
        
        /// <devdoc>
        ///    <para>Gets all copyright notices that apply to the specified file.</para>
        /// </devdoc>
        public string LegalCopyright {
            get {
                return legalCopyright;
            }
        }

        /// <devdoc>
        ///    <para>Gets the trademarks and registered trademarks that apply to the file.</para>
        /// </devdoc>
        public string LegalTrademarks {
            get {
                return legalTrademarks;
            }
        }

        /// <devdoc>
        ///    <para>Gets the name the file was created with.</para>
        /// </devdoc>
        public string OriginalFilename {
            get {
                return originalFilename;
            }
        }

        /// <devdoc>
        ///    <para>Gets information about a private version of the file.</para>
        /// </devdoc>
        public string PrivateBuild {
            get {
                return privateBuild;
            }
        }

        /// <devdoc>
        ///    <para>Gets the build number of the product this file is associated with.</para>
        /// </devdoc>
        public int ProductBuildPart {
            get {
                return productBuild;
            }
        }

        /// <devdoc>
        ///    <para>Gets the major part of the version number for the product this file is associated with.</para>
        /// </devdoc>
        public int ProductMajorPart {
            get {
                return productMajor;
            }
        }

        /// <devdoc>
        ///    <para>Gets the minor part of the version number for the product the file is associated with.</para>
        /// </devdoc>
        public int ProductMinorPart {
            get {
                return productMinor;
            }
        }

        /// <devdoc>
        ///    <para>Gets the name of the product this file is distributed with.</para>
        /// </devdoc>
        public string ProductName {
            get {
                return productName;
            }
        }

        /// <devdoc>
        ///    <para>Gets the private part number of the product this file is associated with.</para>
        /// </devdoc>
        public int ProductPrivatePart {
            get {
                return productPrivate;
            }
        }

        /// <devdoc>
        ///    <para>Gets the version of the product this file is distributed with.</para>
        /// </devdoc>
        public string ProductVersion {
            get {
                return productVersion;
            }
        }

        /// <devdoc>
        ///    <para>Gets the special build information for the file.</para>
        /// </devdoc>
        public string SpecialBuild {
            get {
                return specialBuild;
            }
        }

        private static string ConvertTo8DigitHex(int value) {
            string s = Convert.ToString(value, 16);
            s = s.ToUpper(CultureInfo.InvariantCulture);           
            if (s.Length == 8) {
                return s;
            }
            else {
                StringBuilder b = new StringBuilder(8);
                for (int l = s.Length;l<8; l++) {
                    b.Append("0");
                }
                b.Append(s);
                return b.ToString();
            }
        }
        
        [ResourceExposure(ResourceScope.None)]
        [ResourceConsumption(ResourceScope.Process, ResourceScope.Process)]
        private static NativeMethods.VS_FIXEDFILEINFO GetFixedFileInfo(IntPtr memPtr) {
            IntPtr memRef = IntPtr.Zero;
            int memLen;

            if (UnsafeNativeMethods.VerQueryValue(new HandleRef(null, memPtr), "\\", ref memRef, out memLen)) {
                NativeMethods.VS_FIXEDFILEINFO fixedFileInfo = new NativeMethods.VS_FIXEDFILEINFO();
                Marshal.PtrToStructure(memRef, fixedFileInfo);
                return fixedFileInfo;
            }

            return new NativeMethods.VS_FIXEDFILEINFO();
        }

        private static string GetFileVersionLanguage( IntPtr memPtr ) {
            int langid = GetVarEntry( memPtr ) >> 16;
            
            StringBuilder lang = new StringBuilder( 256 );
            UnsafeNativeMethods.VerLanguageName( langid, lang, lang.Capacity );
            return lang.ToString();
        }
        
        [ResourceExposure(ResourceScope.None)]
        [ResourceConsumption(ResourceScope.Process, ResourceScope.Process)]
        private static string GetFileVersionString(IntPtr memPtr, string name) {
            string data = "";

            IntPtr memRef = IntPtr.Zero;
            int memLen;

            if (UnsafeNativeMethods.VerQueryValue(new HandleRef(null, memPtr), name, ref memRef, out memLen)) {

                if (memRef != IntPtr.Zero) {
                    data = Marshal.PtrToStringAuto(memRef);
                }
            }
            return data;
        }

        [ResourceExposure(ResourceScope.None)]
        [ResourceConsumption(ResourceScope.Process, ResourceScope.Process)]
        private static int GetVarEntry(IntPtr memPtr) {
            IntPtr memRef = IntPtr.Zero;
            int memLen;

            if (UnsafeNativeMethods.VerQueryValue(new HandleRef(null, memPtr), "\\VarFileInfo\\Translation", ref memRef, out memLen)) {
                return(Marshal.ReadInt16(memRef) << 16) + Marshal.ReadInt16((IntPtr)((long)memRef + 2));
            }

            return 0x040904E4;
        }

        // 
        // This function tries to find version informaiton for a specific codepage.
        // Returns true when version information is found.
        //
        private bool GetVersionInfoForCodePage(IntPtr memIntPtr, string codepage) {
            string template = "\\\\StringFileInfo\\\\{0}\\\\{1}";

            companyName = GetFileVersionString(memIntPtr, string.Format(CultureInfo.InvariantCulture, 