/*
 * Argus Client Software.  Tools to read, analyze and manage Argus data.
 * Copyright (c) 2000-2003 QoSient, LLC
 * All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

/*
 * racount - argus data counting program.
 *
 * written by Carter Bullard
 * QoSient, LLC
 *
 */


#include <argus_client.h>


long long totalflowrecords   = 0;
long long totaltcpflowrecs   = 0;
long long totaludpflowrecs   = 0;
long long totalicmpflowrecs  = 0;
long long totalipflowrecs    = 0;
long long totalarpflowrecs   = 0;
long long totalnonipflowrecs = 0;

int RaInitialized = 0;
void
ArgusClientInit ()
{
  if (!(RaInitialized)) {
     RaInitialized++;
  }
}


int RaMonSource = 0;


void
RaParseComplete (int sig)
{
   long long total_count, total_dst_count, total_src_count;
   long long total_bytes, total_dst_bytes, total_src_bytes;
   char *printStr = NULL;

   if (sig >= 0) {
      total_src_count = tcp_src_count + udp_src_count + icmp_src_count +
            ip_src_count + arp_src_count + nonip_src_count;
      total_dst_count = tcp_dst_count + udp_dst_count + icmp_dst_count +
            ip_dst_count + arp_dst_count + nonip_dst_count;
      total_count = total_dst_count + total_src_count;

      total_src_bytes = tcp_src_bytes + udp_src_bytes + icmp_src_bytes +
            ip_src_bytes + arp_src_bytes + nonip_src_bytes;
      total_dst_bytes = tcp_dst_bytes + udp_dst_bytes + icmp_dst_bytes +
            ip_dst_bytes + arp_dst_bytes + nonip_dst_bytes;
      total_bytes = total_dst_bytes + total_src_bytes;

      if (total_count) {
         if (RaMonSource) {
           printf ("%-10s records       total_pkts          in_pkts         out_pkts      ", ArgusProgramName);
           printf ("total_bytes         in_bytes        out_bytes\n");
         } else {
           printf ("%-10s records       total_pkts         src_pkts         dst_pkts      ", ArgusProgramName);
           printf ("total_bytes        src_bytes        dst_bytes\n");
         }

#if defined(HAVE_SOLARIS) || defined(__FreeBSD__) || defined(__NetBSD__) || defined(__OpenBSD__) || defined(CYGWIN)
         printStr = "%7s %10lld %16lld %16lld %16lld %16lld %16lld %16lld\n";
#else
         printStr = "%7s %10Ld %16Ld %16Ld %16Ld %16Ld %16Ld %16Ld\n";
#endif

         if (aflag) {
            if ((tcp_src_count + tcp_dst_count) > 0) {
               printf (printStr, "tcp", totaltcpflowrecs,
                  tcp_dst_count + tcp_src_count, tcp_src_count, tcp_dst_count,
                  tcp_dst_bytes + tcp_src_bytes, tcp_src_bytes, tcp_dst_bytes);
               if ((tcp_src_count + tcp_dst_count) == total_count)
                  goto racount_done;
            }

            if ((udp_src_count + udp_dst_count) > 0) {
               printf (printStr, "udp", totaludpflowrecs,
                  udp_dst_count + udp_src_count, udp_src_count, udp_dst_count,
                  udp_dst_bytes + udp_src_bytes, udp_src_bytes, udp_dst_bytes);
               if ((udp_src_count + udp_dst_count) == total_count)
                  goto racount_done;
            }

            if ((icmp_src_count + icmp_dst_count) > 0) {
               printf (printStr, "icmp", totalicmpflowrecs,
                  icmp_dst_count + icmp_src_count, icmp_src_count, icmp_dst_count,
                  icmp_dst_bytes + icmp_src_bytes, icmp_src_bytes, icmp_dst_bytes);
               if ((icmp_src_count + icmp_dst_count) == total_count)
                  goto racount_done;
            }

            if ((ip_src_count + ip_dst_count) > 0) {
               printf (printStr, "ip", totalipflowrecs,
                  ip_dst_count + ip_src_count, ip_src_count, ip_dst_count,
                  ip_dst_bytes + ip_src_bytes, ip_src_bytes, ip_dst_bytes);
               if ((ip_src_count + ip_dst_count) == total_count)
                  goto racount_done;
            }

            if ((arp_src_count + arp_dst_count) > 0) {
               printf (printStr, "arp", totalarpflowrecs,
                  arp_dst_count + arp_src_count, arp_src_count, arp_dst_count,
                  arp_dst_bytes + arp_src_bytes, arp_src_bytes, arp_dst_bytes);
               if ((arp_src_count + arp_dst_count) == total_count)
                  goto racount_done;
            }

            if ((nonip_src_count + nonip_dst_count) > 0) {
               printf (printStr, "non-ip", totalnonipflowrecs,
                  nonip_dst_count + nonip_src_count, nonip_src_count, nonip_dst_count,
                  nonip_dst_bytes + nonip_src_bytes, nonip_src_bytes, nonip_dst_bytes);
               if ((nonip_src_count + nonip_dst_count) == total_count)
                  goto racount_done;
            }
         }

         printf (printStr, "sum", totalflowrecords,
               total_count, total_src_count, total_dst_count,
               total_bytes, total_src_bytes, total_dst_bytes);

      } else
         printf ("No data seen.\n");

racount_done:

      fflush (stdout);
      _exit(0);
   }
}


void ArgusClientTimeout ()
{
}

void
parse_arg (int argc, char**argv)
{
}


void
usage ()
{
   extern char version[];

   fprintf (stderr, "Racount Version %s\n", version);
   fprintf (stderr, "usage: %s \n", ArgusProgramName);
   fprintf (stderr, "usage: %s [ra-options] [- filter-expression]\n", ArgusProgramName);

   fprintf (stderr, "ra-options: -b               dump packet-matching code.\n");
   fprintf (stderr, "            -C               treat the remote source as a Cisco Netflow source.\n");
   fprintf (stderr, "            -D <level>       specify debug level\n");
   fprintf (stderr, "            -F <conffile>    read configuration from <conffile>.\n");
   fprintf (stderr, "            -h               print help.\n");
   fprintf (stderr, "            -P <portnum>     specify remote argus <portnum> (tcp/561).\n");
   fprintf (stderr, "            -r <file>        read argus data <file>. '-' denotes stdin.\n");
   fprintf (stderr, "            -S <host[:port]> specify remote argus <host> and optional port number.\n");
   fprintf (stderr, "            -t <timerange>   specify <timerange> for reading records.\n");
   fprintf (stderr, "                    format:  timeSpecification[-timeSpecification]\n");
   fprintf (stderr, "                             timeSpecification: [mm/dd[/yy].]hh[:mm[:ss]]\n");
   fprintf (stderr, "                                                 mm/dd[/yy]\n");
   fprintf (stderr, "                                                 -%%d{yMhdms}\n");
#ifdef ARGUS_SASL
   fprintf (stderr, "            -U <user/auth>   specify <user/auth> authentication information.\n");
#endif
   fprintf (stderr, "            -T <secs>        attach to remote server for T seconds.\n");
   exit(1);
}


void
RaProcessRecord (struct ArgusRecord *argus)
{
   totalflowrecords++;

   if (argus->ahdr.type & ARGUS_MAR) {
      RaProcessManRecord (argus);

   } else {
      if (argus->ahdr.type & ARGUS_RMON)
         RaMonSource++;
      else
         RaMonSource = 0;

      switch (argus->ahdr.status & 0xFFFF) {
         case ETHERTYPE_IP:
            switch (argus->argus_far.flow.ip_flow.ip_p) {
               case IPPROTO_TCP:
                  RaProcessTCPRecord (argus);
                  break;

               case IPPROTO_UDP:
                  RaProcessUDPRecord (argus);
                  break;

               case IPPROTO_ICMP:
                  RaProcessICMPRecord (argus);
                  break;

               default:
                  RaProcessIPRecord (argus);
                  break;
            }
            break;

         case ETHERTYPE_ARP:
         case ETHERTYPE_REVARP:
            RaProcessARPRecord (argus);
            break;

         default:
            RaProcessNonIPRecord (argus);
            break;
      }
   }
}


void
RaProcessManRecord (struct ArgusRecord *argus)
{
}


void
RaProcessTCPRecord (struct ArgusRecord *argus)
{
   totaltcpflowrecs++;

   tcp_src_count += argus->argus_far.src.count;
   tcp_dst_count += argus->argus_far.dst.count;

   if (Aflag) {
      tcp_src_bytes += argus->argus_far.src.appbytes;
      tcp_dst_bytes += argus->argus_far.dst.appbytes;
   } else {
      tcp_src_bytes += argus->argus_far.src.bytes;
      tcp_dst_bytes += argus->argus_far.dst.bytes;
   }
}


void
RaProcessICMPRecord (struct ArgusRecord *argus)
{
   totalicmpflowrecs++;

   icmp_src_count += argus->argus_far.src.count;
   icmp_dst_count += argus->argus_far.dst.count;

   if (Aflag) {
      icmp_src_bytes += argus->argus_far.src.appbytes;
      icmp_dst_bytes += argus->argus_far.dst.appbytes;
   } else {
      icmp_src_bytes += argus->argus_far.src.bytes;
      icmp_dst_bytes += argus->argus_far.dst.bytes;
   }
}


void
RaProcessUDPRecord (struct ArgusRecord *argus)
{
   totaludpflowrecs++;

   udp_src_count += argus->argus_far.src.count;
   udp_dst_count += argus->argus_far.dst.count;

   if (Aflag) {
      udp_src_bytes += argus->argus_far.src.appbytes;
      udp_dst_bytes += argus->argus_far.dst.appbytes;
   } else {
      udp_src_bytes += argus->argus_far.src.bytes;
      udp_dst_bytes += argus->argus_far.dst.bytes;
   }
}


void
RaProcessIPRecord (struct ArgusRecord *argus)
{
   totalipflowrecs++;

   ip_src_count += argus->argus_far.src.count;
   ip_dst_count += argus->argus_far.dst.count;

   if (Aflag) {
      ip_src_bytes += argus->argus_far.src.appbytes;
      ip_dst_bytes += argus->argus_far.dst.appbytes;
   } else {
      ip_src_bytes += argus->argus_far.src.bytes;
      ip_dst_bytes += argus->argus_far.dst.bytes;
   }
}


void
RaProcessARPRecord (struct ArgusRecord *argus)
{
   totalarpflowrecs++;

   arp_src_count += argus->argus_far.src.count;
   arp_dst_count += argus->argus_far.dst.count;

   if (Aflag) {
      arp_src_bytes += argus->argus_far.src.appbytes;
      arp_dst_bytes += argus->argus_far.dst.appbytes;
   } else {
      arp_src_bytes += argus->argus_far.src.bytes;
      arp_dst_bytes += argus->argus_far.dst.bytes;
   }
}


void
RaProcessNonIPRecord (struct ArgusRecord *argus)
{
   totalnonipflowrecs++;

   nonip_src_count += argus->argus_far.src.count;
   nonip_dst_count += argus->argus_far.dst.count;

   if (Aflag) {
      nonip_src_bytes += argus->argus_far.src.appbytes;
      nonip_dst_bytes += argus->argus_far.dst.appbytes;
   } else {
      nonip_src_bytes += argus->argus_far.src.bytes;
      nonip_dst_bytes += argus->argus_far.dst.bytes;
   }
}

int RaSendArgusRecord(struct ArgusRecordStore *argus) {return 0;}
