# -*- mode: python; coding: utf-8 -*-
#
# Pigment Python tools unit tests
#
# Copyright © 2006, 2007, 2008 Fluendo Embedded S.L.
#
# This library is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public
# License as published by the Free Software Foundation; either
# version 2 of the License, or (at your option) any later version.
#
# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public
# License along with this library; if not, write to the
# Free Software Foundation, Inc., 59 Temple Place - Suite 330,
# Boston, MA 02111-1307, USA.

import unittest
import sys
import gobject

from pgm.graph.group import Group
from pgm.graph.image import Image
from pgm.widgets.list import List
import pgm
 
class TestList(unittest.TestCase):

    def test_image(self):
        canvas = pgm.Canvas()

        l = List(canvas, pgm.DRAWABLE_MIDDLE)
        img = Image()

        pygobj_has_toggle_notify = gobject.pygobject_version >= (2, 13, 0)
        if pygobj_has_toggle_notify:
            initial_refcount = 2
        else:
            initial_refcount = 3
        expected_refcount = initial_refcount

        self.assertEqual(sys.getrefcount(l), initial_refcount)
        self.assertEqual(sys.getrefcount(img), initial_refcount)

        # adding img directly to the list _as a Group_
        l.add(img)
        if pygobj_has_toggle_notify:
            # here one reference is taken by the group, another by the canvas
            expected_refcount += 2
        else:
            # just a reference by the group
            expected_refcount += 1
        self.assertEqual (sys.getrefcount(img), expected_refcount)

        # trying to add img again to the list _as a Group_
        l.add(img)
        self.assertEqual(sys.getrefcount(img), expected_refcount)

        # removing img from the list
        l.remove(img)
        if pygobj_has_toggle_notify:
            expected_refcount -= 2
        else:
            expected_refcount -= 1
        self.assertEqual(sys.getrefcount(img), expected_refcount)

        # same but adding img to the canvas before adding it to the list _as a
        # Group_
        canvas.add(pgm.DRAWABLE_MIDDLE, img)
        if pygobj_has_toggle_notify:
            # g_object_ref is called on the object here, so pygobject calls
            # Py_INCREF on img
            expected_refcount += 1
        self.assertEqual(sys.getrefcount(img), expected_refcount)

        l.add(img)
        expected_refcount += 1
        self.assertEqual(sys.getrefcount(img), expected_refcount)
        l.remove(img)
        if pygobj_has_toggle_notify:
            # a ref is also removed from the canvas here
            expected_refcount -= 2
        else:
            expected_refcount -= 1
        self.assertEqual(sys.getrefcount(img), expected_refcount)

        canvas.remove(img)
        self.assertEqual(sys.getrefcount(img), initial_refcount)

        # adding img directly to the list _as a List_

        l.append(img)
        # 4 references added: one for the Group, one for the List, two for the
        # implicitely animated object (implicit + one modifier)
        expected_refcount += 4
        if pygobj_has_toggle_notify:
            # the Group takes two refs here (one in the canvas)
            expected_refcount += 1

        self.assertEqual(sys.getrefcount(img), expected_refcount)
        self.assertEqual(len(l.widgets), 1)
        self.assertEqual(len(l.animated_widgets), 1)

        # trying to add img again to the list _as a List_
        l.append(img)
        self.assertEqual(sys.getrefcount(img), expected_refcount)
        self.assertEqual(len(l.widgets), 1)
        self.assertEqual(len(l.animated_widgets), 1)

        # removing img from the list
        l.remove(img)
        self.assertEqual(len(l.widgets), 0)
        self.assertEqual(len(l.animated_widgets), 0)
        self.assertEqual(sys.getrefcount(img), initial_refcount)

    def test_group(self):
        canvas = pgm.Canvas()
        l = List(canvas, pgm.DRAWABLE_MIDDLE)
        g = Group(canvas, pgm.DRAWABLE_MIDDLE)


        pygobj_has_toggle_notify = gobject.pygobject_version >= (2, 13, 0)
        if pygobj_has_toggle_notify:
            g_initial_refcount = 2
            l_initial_refcount = 2
        else:
            g_initial_refcount = 3
            l_initial_refcount = 3

        self.assertEqual(sys.getrefcount(l), l_initial_refcount)
        self.assertEqual(sys.getrefcount(g), g_initial_refcount)

        # adding group directly to the list _as a List_
        l.append(g)
        self.assertEqual(sys.getrefcount(g), g_initial_refcount+6)
        self.assertEqual(len(l.widgets), 1)
        self.assertEqual(len(l.animated_widgets), 1)

        # removing group from the list
        l.remove(g)
        self.assertEqual(len(l.widgets), 0)
        self.assertEqual(len(l.animated_widgets), 0)
        self.assertEqual(sys.getrefcount(g), g_initial_refcount)

    def test_multiple(self):
        canvas = pgm.Canvas()
        l = List(canvas, pgm.DRAWABLE_MIDDLE)
        g = Group(canvas, pgm.DRAWABLE_MIDDLE)
        img = Image()
        pygobj_has_toggle_notify = gobject.pygobject_version >= (2, 13, 0)
        if pygobj_has_toggle_notify:
            img_initial_refcount = 2
            g_initial_refcount = g_expected_refcount = 2
            l_initial_refcount = 2
        else:
            img_initial_refcount = 3
            g_initial_refcount = g_expected_refcount = 3
            l_initial_refcount = 3
        img_expected_refcount = img_initial_refcount

        self.assertEqual(sys.getrefcount(l), l_initial_refcount)
        self.assertEqual(sys.getrefcount(g), g_initial_refcount)
        self.assertEqual(sys.getrefcount(img), img_initial_refcount)

        # adding group directly to the list _as a List_
        l.append(g)
        l.append(img)
        if pygobj_has_toggle_notify:
            img_expected_refcount += 5
        else:
            img_expected_refcount += 4
        g_expected_refcount += 6
        self.assertEqual(sys.getrefcount(img), img_expected_refcount)
        self.assertEqual(sys.getrefcount(g), g_expected_refcount)
        self.assertEqual(len(l.widgets), 2)
        self.assertEqual(len(l.animated_widgets), 2)

        # removing img from the list
        l.remove(img)
        if pygobj_has_toggle_notify:
            img_expected_refcount -= 5
        else:
            img_expected_refcount -= 4
        self.assertEqual(len(l.widgets), 1)
        self.assertEqual(len(l.animated_widgets), 1)
        self.assertEqual(sys.getrefcount(img), img_expected_refcount)

        # adding img to the group
        g.add(img)
        if pygobj_has_toggle_notify:
            img_expected_refcount += 2
        else:
            img_expected_refcount += 1
        self.assertEqual(len(l.widgets), 1)
        self.assertEqual(len(l.animated_widgets), 1)
        self.assertEqual(sys.getrefcount(img), img_expected_refcount)
        self.assertEqual(sys.getrefcount(g), g_expected_refcount)

        # removing group from the list
        l.remove(g)
        g.remove(img)
        g_expected_refcount -= 6
        img_expected_refcount -= 1
        self.assertEqual(len(l.widgets), 0)
        self.assertEqual(len(l.animated_widgets), 0)
        self.assertEqual(sys.getrefcount(l), l_initial_refcount)
        self.assertEqual(sys.getrefcount(g), g_initial_refcount)
        self.assertEqual(sys.getrefcount(img), img_initial_refcount)
