/*	JTreeTable


Copyright 1997-2000 Sun Microsystems, Inc. All Rights Reserved.

Redistribution and use in source and binary forms, with or
without modification, are permitted provided that the following
conditions are met:

- Redistributions of source code must retain the above copyright
  notice, this list of conditions and the following disclaimer. 
  
- Redistribution in binary form must reproduce the above
  copyright notice, this list of conditions and the following
  disclaimer in the documentation and/or other materials
  provided with the distribution. 
  
Neither the name of Sun Microsystems, Inc. or the names of
contributors may be used to endorse or promote products derived
from this software without specific prior written permission.  

This software is provided "AS IS," without a warranty of any
kind. ALL EXPRESS OR IMPLIED CONDITIONS, REPRESENTATIONS AND
WARRANTIES, INCLUDING ANY IMPLIED WARRANTY OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE OR NON-INFRINGEMENT, ARE HEREBY
EXCLUDED. SUN AND ITS LICENSORS SHALL NOT BE LIABLE FOR ANY
DAMAGES OR LIABILITIES SUFFERED BY LICENSEE AS A RESULT OF OR
RELATING TO USE, MODIFICATION OR DISTRIBUTION OF THIS SOFTWARE OR
ITS DERIVATIVES. IN NO EVENT WILL SUN OR ITS LICENSORS BE LIABLE 
FOR ANY LOST REVENUE, PROFIT OR DATA, OR FOR DIRECT, INDIRECT,   
SPECIAL, CONSEQUENTIAL, INCIDENTAL OR PUNITIVE DAMAGES, HOWEVER  
CAUSED AND REGARDLESS OF THE THEORY OF LIABILITY, ARISING OUT OF 
THE USE OF OR INABILITY TO USE THIS SOFTWARE, EVEN IF SUN HAS 
BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.

You acknowledge that this software is not designed, licensed or
intended for use in the design, construction, operation or
maintenance of any nuclear facility.


Modfied at:

	Planetary Image Research Laboratory
	University of Arizona
	Tucson, AZ 85721-0092

By:

	Bradford Castalia

PIRL CVS ID: JTreeTable.java,v 1.4 2007/04/20 05:01:40 castalia Exp
*/
package	PIRL.TreeTable;

import javax.swing.*;
import javax.swing.border.*;
import javax.swing.event.*;
import javax.swing.tree.*;
import javax.swing.table.*;

import java.awt.Dimension;
import java.awt.Color;
import java.awt.Component;
import java.awt.Graphics;
import java.awt.Rectangle;
import java.awt.event.MouseEvent;
import java.awt.event.InputEvent;

import java.util.EventObject;

/**
 * This example shows how to create a simple JTreeTable component, 
 * by using a JTree as a renderer (and editor) for the cells in a 
 * particular column in the JTable.  
 *
 * @version 1.2 10/27/98
 *
 * @author Philip Milne
 * @author Scott Violet
 */
public class JTreeTable
	extends JTable
{
/**	A subclass of JTree.
*/
protected TreeTableCellRenderer		The_Tree;


private static final int
	DEBUG_OFF				= 0,
	DEBUG_CONSTRUCTORS		= 1 << 0,
	DEBUG_ACCESSORS			= 1 << 1,
	DEBUG_MANIPULATORS		= 1 << 2,
	DEBUG_CELL_EDITOR		= 1 << 3,
	DEBUG_ALL				= -1,

	DEBUG					= DEBUG_OFF;


public JTreeTable
	(
	TreeTableModel	treeTableModel
	)
{
super();

// Create the tree. It will be used as a renderer and editor. 
The_Tree = new TreeTableCellRenderer (treeTableModel);

// Install a tableModel representing the visible rows in the tree. 
super.setModel (new TreeTableModelAdapter (treeTableModel, The_Tree));

// Force the JTable and JTree to share their row selection models. 
ListToTreeSelectionModelWrapper
	selectionWrapper = new ListToTreeSelectionModelWrapper ();
The_Tree.setSelectionModel (selectionWrapper);
setSelectionModel (selectionWrapper.getListSelectionModel ()); 

// Install the tree editor renderer and editor. 
setDefaultRenderer (TreeTableModel.class, The_Tree); 
setDefaultEditor (TreeTableModel.class, new TreeTableCellEditor ());

// No grid.
setShowGrid (false);

// No intercell spacing
setIntercellSpacing (new Dimension (0, 0));	

if (The_Tree.getRowHeight () < 1)
	// Metal looks better like this.
	setRowHeight (18);
}

/**	Gets the Object at a row of the Tree.
*/
public Object getTreeNode
	(
	int			row
	)
{return ((TreeTableModelAdapter)getModel ()).nodeForRow (row);}

/**	Overridden to message super and forward the method to the tree.

Since the tree is not actually in the component hierarchy it will
never receive this unless we forward it in this manner.
*/
public void updateUI ()
{
super.updateUI ();
if (The_Tree != null)
	{
	The_Tree.updateUI ();
	/*
		Do this so that the editor is referencing the current renderer
	    from the tree. The renderer can potentially change each time
	    laf changes.
	*/
	setDefaultEditor (TreeTableModel.class, new TreeTableCellEditor ());
	}
// Use the tree's default foreground and background colors in the
// table. 
LookAndFeel.installColorsAndFont
	(
	this,
	"Tree.background",
	"Tree.foreground",
	"Tree.font"
	);
}

/** Workaround for BasicTableUI anomaly.

	Make sure the UI never tries to paint the editor. The UI currently
	uses different techniques to paint the renderers and editors and
	overriding setBounds() below is not the right thing to do for an
	editor. Returning -1 for the editing row in this case, ensures the
	editor is never painted. 
*/
public int getEditingRow ()
{
return (getColumnClass (editingColumn) == TreeTableModel.class) ?
	-1 : editingRow;  
}

/**	Gets the actual row that is editing as <code>getEditingRow</code>
	will always return -1.
*/
private int realEditingRow ()
{return editingRow;}

/**	This is overridden to invoke super's implementation, and then,
	if the receiver is editing a Tree column, the editor's bounds is
	reset. The reason we have to do this is because JTable doesn't
	think the table is being edited, as <code>getEditingRow</code> returns
	-1, and therefore doesn't automatically resize the editor for us.
*/
public void sizeColumnsToFit
	(
	int			resizingColumn
	)
{ 
super.sizeColumnsToFit (resizingColumn);
if (getEditingColumn () != -1 &&
	getColumnClass (editingColumn) == TreeTableModel.class)
	{
	Rectangle
		cellRect = getCellRect (realEditingRow (), getEditingColumn (), false);
	Component
		component = getEditorComponent ();
	component.setBounds (cellRect);
	component.validate();
	}
}

/**	Overridden to pass the new rowHeight to the tree.
*/
public void setRowHeight
	(
	int			rowHeight
	)
{ 
super.setRowHeight (rowHeight); 
if (The_Tree != null &&
	The_Tree.getRowHeight () != rowHeight)
	The_Tree.setRowHeight (getRowHeight ()); 
}

/**	Overridden to invoke repaint for the particular location if
	the column contains the tree. This is done as the tree editor does
	not fill the bounds of the cell, we need the renderer to paint
	the tree in the background, and then draw the editor over it.
*/
public boolean editCellAt
	(
	int			row,
	int			column,
	EventObject	event
	)
{
boolean
	result = super.editCellAt (row, column, event);
if (result &&
	getColumnClass (column) == TreeTableModel.class)
	repaint (getCellRect (row, column, false));
return result;
}

/**	Get the tree that is being shared between the model.
*/
public JTree getTree ()
{return The_Tree;}

/*------------------------------------------------------------------------------
*/
/**	A TreeCellRenderer that displays a JTree.
*/
public class TreeTableCellRenderer
	extends JTree
	implements TableCellRenderer
{
/**	Last table/tree row asked to render.
*/
protected int visibleRow;

/**	Border to draw around the tree.

	If this is non-null, it will be painted.
*/
protected Border highlightBorder;


public TreeTableCellRenderer
	(
	TreeModel	model
	)
{super (model);}

/**	Set the colors of the Tree's renderer to match that of the table.
*/
public void updateUI ()
{
if ((DEBUG & DEBUG_MANIPULATORS) != 0)
	System.out.println (">>> JTreeTable.updateUI");
super.updateUI ();

// Make the tree's cell renderer use the table's cell selection colors. 
TreeCellRenderer
	renderer = getCellRenderer ();
if (renderer instanceof DefaultTreeCellRenderer)
	{
	DefaultTreeCellRenderer
		default_renderer = ((DefaultTreeCellRenderer)renderer);

	/*
		For 1.1 uncomment this, 1.2 has a bug that will cause an
		exception to be thrown if the border selection color is null.
	*/
	// default_renderer.setBorderSelectionColor (null);
	if ((DEBUG & DEBUG_MANIPULATORS) != 0)
		System.out.println
			("      UIManager.getColor (\"Table.selectionForeground\") = "
				+ UIManager.getColor ("Table.selectionForeground") + '\n'
			+"      UIManager.getColor (\"Table.selectionBackground\") = "
				+ UIManager.getColor ("Table.selectionBackground"));
	default_renderer.setTextSelectionColor
		(UIManager.getColor ("Table.selectionForeground"));
	default_renderer.setBackgroundSelectionColor
		(UIManager.getColor ("Table.selectionBackground"));
	}
if ((DEBUG & DEBUG_MANIPULATORS) != 0)
	System.out.println ("<<< JTreeTable.updateUI");
}

/**	Sets the row height of the tree,
	and forward the row height to the table.
*/
public void setRowHeight
	(
	int			rowHeight
	)
{ 
if (rowHeight > 0)
	{
	super.setRowHeight (rowHeight); 
	if (JTreeTable.this != null &&
		JTreeTable.this.getRowHeight () != rowHeight)
		JTreeTable.this.setRowHeight (getRowHeight ()); 
	}
}

/**	Set the height to match that of the JTable.
*/
public void setBounds
	(
	int			x,
	int			y,
	int			w,
	int			h
	)
{super.setBounds (x, 0, w, JTreeTable.this.getHeight ());}

/**	Translate the graphics such that
	the last visible row will be drawn at 0,0.
*/
public void paint
	(
	Graphics	graphics
	)
{
if ((DEBUG & DEBUG_MANIPULATORS) != 0)
	System.out.println (">>> JTreeTable.paint");
graphics.translate (0, -visibleRow * getRowHeight ());
super.paint (graphics);

//	Draw the Table border if we have focus.
if (highlightBorder != null)
	highlightBorder.paintBorder (this, graphics,
		0, visibleRow * getRowHeight (), getWidth (), getRowHeight ());

//	Fill in a cell that has an oversized row height.
if (JTreeTable.this.getRowHeight (visibleRow) > getRowHeight ())
	{
	if ((DEBUG & DEBUG_MANIPULATORS) != 0)
		System.out.println
			("      Visible row " + visibleRow + " height = "
				+ JTreeTable.this.getRowHeight (visibleRow)
				+ " > normal row height " + getRowHeight () + '\n'
			+"      Width = " + getWidth () + '\n'
			+"      graphics.setColor (UIManager.getColor"
				+ " (\"Table.selectionBackground\") = "
				+ UIManager.getColor ("Table.selectionBackground") + '\n'
			+"      graphics.fillRect (0, "
				+ visibleRow * getRowHeight() + getRowHeight ()
				+ ", " + getWidth ()
				+ ", " + JTreeTable.this.getRowHeight (visibleRow) +')');
	graphics.setColor (UIManager.getColor ("Table.selectionBackground"));
	graphics.fillRect
		(0, visibleRow * getRowHeight() + getRowHeight (), 
		getWidth (), JTreeTable.this.getRowHeight (visibleRow));
	}
if ((DEBUG & DEBUG_MANIPULATORS) != 0)
	System.out.println ("<<< JTreeTable.paint");
}

/**	Update the visible row.
*/
public Component getTableCellRendererComponent
	(
	JTable		table,
	Object		value,
	boolean		isSelected,
	boolean		hasFocus,
	int			row,
	int			column
	)
{
if ((DEBUG & DEBUG_MANIPULATORS) != 0)
	System.out.println (">>> JTreeTable.getTableCellRendererComponent:"
		+ " row " + row + ", column " + column);
Color
	background,
	foreground;

if (isSelected)
	{
	background = table.getSelectionBackground ();
	foreground = table.getSelectionForeground ();
	if ((DEBUG & DEBUG_MANIPULATORS) != 0)
		System.out.println
			("      isSelected true\n"
			+"      table.getSelectionBackground () = " + background + '\n'
			+"      table.getSelectionForeground () = " + foreground);
	}
else
	{
	background = table.getBackground ();
	foreground = table.getForeground ();
	if ((DEBUG & DEBUG_MANIPULATORS) != 0)
		System.out.println
			("      isSelected false\n"
			+"      table.getBackground () = " + background + '\n'
			+"      table.getForeground () = " + foreground);
	}
highlightBorder = null;
if (realEditingRow () == row &&
	getEditingColumn () == column)
	{
	background = UIManager.getColor ("Table.focusCellBackground");
	foreground = UIManager.getColor ("Table.focusCellForeground");
	if ((DEBUG & DEBUG_MANIPULATORS) != 0)
		System.out.println
			("      Editing this row and column\n"
			+"      UIManager.getColor (\"Table.focusCellBackground\") = "
				+ background + '\n'
			+"      UIManager.getColor (\"Table.focusCellForeground\") = "
				+ foreground);
	}
else if (hasFocus)
	{
	highlightBorder = UIManager.getBorder ("Table.focusCellHighlightBorder");
	if (isCellEditable (row, column))
		{
		background = UIManager.getColor ("Table.focusCellBackground");
		foreground = UIManager.getColor ("Table.focusCellForeground");
		if ((DEBUG & DEBUG_MANIPULATORS) != 0)
			System.out.println
				("      hasFocus true and cell is editable\n"
				+"      UIManager.getColor (\"Table.focusCellBackground\") = "
					+ background + '\n'
				+"      UIManager.getColor (\"Table.focusCellForeground\") = "
					+ foreground);
		}
	}
visibleRow = row;
setBackground (background);
if ((DEBUG & DEBUG_MANIPULATORS) != 0)
	System.out.println
		("      Visible row: " + row + '\n'
		+"      Background = " + background);

TreeCellRenderer renderer = getCellRenderer ();
if (renderer instanceof DefaultTreeCellRenderer)
	{
	DefaultTreeCellRenderer
		default_renderer = ((DefaultTreeCellRenderer)renderer); 
	if (isSelected)
		{
		default_renderer.setTextSelectionColor (foreground);
		default_renderer.setBackgroundSelectionColor (background);
		if ((DEBUG & DEBUG_MANIPULATORS) != 0)
			System.out.println
				("      default_renderer.setBackgroundSelectionColor = "
					+ background + '\n'
				+"      default_renderer.setTextSelectionColor       = "
					+ foreground);
		}
	else
		{
		default_renderer.setTextNonSelectionColor (foreground);
		default_renderer.setBackgroundNonSelectionColor (background);
		if ((DEBUG & DEBUG_MANIPULATORS) != 0)
			System.out.println
				("      default_renderer.setBackgroundNonSelectionColor = "
					+ background + '\n'
				+"      default_renderer.setTextNonSelectionColor       = "
					+ foreground);
		}
	}
if ((DEBUG & DEBUG_MANIPULATORS) != 0)
	System.out.println ("<<< JTreeTable.getTableCellRendererComponent");
return this;
}

}	//	class TreeTableCellRenderer.

/*------------------------------------------------------------------------------
*/
/**	TreeTableCellEditor implementation.

	An editor that can be used to edit the tree column. This extends
	DefaultCellEditor and uses a JTextField (actually,
	TreeTableTextField) to perform the actual editing.
<p>
	To support editing of the tree column we can not make the tree
	editable. The reason this doesn't work is that you can not use the
	same component for editing and renderering. The table may have the
	need to paint cells, while a cell is being edited. If the same
	component were used for the rendering and editing the component
	would be moved around, and the contents would change. When editing,
	this is undesirable, the contents of the text field must stay the
	same, including the caret blinking, and selections persisting. For
	this reason the editing is done via a TableCellEditor.
<p>
	Another interesting thing to be aware of is how tree positions its
	render and editor. The render/editor is responsible for drawing the
	icon indicating the type of node (leaf, branch...). The tree is
	responsible for drawing any other indicators, perhaps an additional
	+/- sign, or lines connecting the various nodes. So, the renderer
	is positioned based on depth. On the other hand, table always makes
	its editor fill the contents of the cell. To get the allusion that
	the table cell editor is part of the tree, we don't want the table
	cell editor to fill the cell bounds. We want it to be placed in the
	same manner as tree places it editor, and have table message the
	tree to paint any decorations the tree wants. Then, we would only
	have to worry about the editing part. The approach taken here is to
	determine where tree would place the editor, and to override the
	<code>reshape</code> method in the JTextField component to nudge
	the textfield to the location tree would place it. Since JTreeTable
	will paint the tree behind the editor everything should just work.
	So, that is what we are doing here. Determining of the icon
	position will only work if the TreeCellRenderer is an instance of
	DefaultTreeCellRenderer. If you need custom TreeCellRenderers, that
	don't descend from DefaultTreeCellRenderer,  and you want to
	support editing in JTreeTable, you will have to do something
	similiar.
*/
public class TreeTableCellEditor
	extends DefaultCellEditor
{
public TreeTableCellEditor ()
{super (new TreeTableTextField ());}


/**	Overridden to determine an offset where the tree would place the
	editor.

	The offset is determined from the <code>getRowBounds</code> JTree
	method, and additionally from the icon DefaultTreeCellRenderer will
	use. The offset is then set on the TreeTableTextField component
	created in the constructor, and returned.
*/
public Component getTableCellEditorComponent
	(
	JTable		table,
	Object		value,
	boolean		isSelected,
	int			row,
	int			column
	)
{
if ((DEBUG & DEBUG_CELL_EDITOR) != 0)
	System.out.println
		(">>> JTreeTable.TreeTableCellEditor.getTableCellEditorComponent:\n"
		+"    isSelected " + isSelected
			+ ", row " + row + ", column " + column);
Component
	component = super.getTableCellEditorComponent
		(table, value, isSelected, row, column);
JTree
	tree = getTree ();
int
	offsetRow =  tree.isRootVisible () ? row : row - 1;
if (offsetRow < 0)
	offsetRow = 0;
Rectangle
	bounds = tree.getRowBounds (offsetRow);
if ((DEBUG & DEBUG_CELL_EDITOR) != 0)
	System.out.println
		("    bounds at offsetRow " + offsetRow + " = " + bounds);
int
	offset = bounds.x;
TreeCellRenderer
	renderer = tree.getCellRenderer ();
if (renderer instanceof DefaultTreeCellRenderer)
	{
	Object
		node = tree.getPathForRow (offsetRow).getLastPathComponent ();
	Icon
		icon;
	if (tree.getModel ().isLeaf (node))
		icon = ((DefaultTreeCellRenderer)renderer).getLeafIcon ();
	else if (The_Tree.isExpanded (offsetRow))
		icon = ((DefaultTreeCellRenderer)renderer).getOpenIcon ();
	else
		icon = ((DefaultTreeCellRenderer)renderer).getClosedIcon ();
	if (icon != null)
		offset += ((DefaultTreeCellRenderer)renderer).getIconTextGap () +
			icon.getIconWidth ();
	}
((TreeTableTextField)getComponent ()).offset = offset;
if ((DEBUG & DEBUG_CELL_EDITOR) != 0)
	System.out.println
		("<<< JTreeTable.TreeTableCellEditor.getTableCellEditorComponent:");
return component;
}

/**	Overridden to forward the event to the tree.

	Returns true if the click count >= 3, or the event is null.
*/
public boolean isCellEditable
	(
	EventObject	event
	)
{
if (event instanceof MouseEvent)
	{
	MouseEvent mouse_event = (MouseEvent)event;
	/*
		If the modifiers are not 0 (or the left mouse button), tree may
		try and toggle the selection, and table will then try and
		toggle, resulting in the selection remaining the same. To avoid
		this, we only dispatch when the modifiers are 0 (or the left
		mouse button).
	*/
	if (mouse_event.getModifiers () == 0 ||
		mouse_event.getModifiers() == InputEvent.BUTTON1_MASK)
		{
		for (int counter = getColumnCount () - 1;
				 counter >= 0;
				 counter--)
			{
			if (getColumnClass (counter) == TreeTableModel.class)
				{
				JTreeTable.this.The_Tree.dispatchEvent (new MouseEvent
					(
					JTreeTable.this.The_Tree,
					mouse_event.getID (),
					mouse_event.getWhen (),
					mouse_event.getModifiers (),
					mouse_event.getX () - getCellRect (0, counter, true).x,
					mouse_event.getY (),
					mouse_event.getClickCount (),
					mouse_event.isPopupTrigger ()
					));
				break;
				}
		    }
		}
	if (mouse_event.getClickCount () >= 3)
		return true;
	return false;
	}
if (event == null)
	return true;
return false;
}

}	//	class TreeTableCellEditor.

/*------------------------------------------------------------------------------
*/
/**	Component used by TreeTableCellEditor.

	The only thing this does is to override the <code>reshape</code>
	method, and to ALWAYS make the x location be <code>offset</code>.
*/
static class TreeTableTextField
	extends JTextField
{
public int				offset;

public void reshape
	(
	int		x,
	int		y,
	int		w,
	int		h
	)
{
int
	newX = Math.max(x, offset);
super.reshape (newX, y, w - (newX - x), h);
}

}	//	class TreeTableTextField.

/*------------------------------------------------------------------------------
*/
/**	ListToTreeSelectionModelWrapper extends DefaultTreeSelectionModel
	to listen for changes in the ListSelectionModel it maintains. Once
	a change in the ListSelectionModel happens, the paths are updated
	in the DefaultTreeSelectionModel.
*/
class ListToTreeSelectionModelWrapper
	extends DefaultTreeSelectionModel
{
/**	Set to true when we are updating the ListSelectionModel.
*/
protected boolean		updatingListSelectionModel;

public ListToTreeSelectionModelWrapper ()
{
super ();
getListSelectionModel ()
	.addListSelectionListener (createListSelectionListener ());
}

/**	Returns the list selection model.

	ListToTreeSelectionModelWrapper listens for changes to this model
	and updates the selected paths accordingly.
*/
ListSelectionModel getListSelectionModel ()
{return listSelectionModel;}

/**	Sets <code>updatingListSelectionModel</code> and message super.

	This is the only place DefaultTreeSelectionModel alters the
	ListSelectionModel.
*/
public void resetRowSelection ()
{
if (! updatingListSelectionModel)
	{
	updatingListSelectionModel = true;
	try {super.resetRowSelection ();}
	finally {updatingListSelectionModel = false;}
	}
/*
	Notice how we don't message super if updatingListSelectionModel is
	true. If updatingListSelectionModel is true, it implies the
	ListSelectionModel has already been updated and the paths are the
	only thing that needs to be updated.
*/
}

/**	Creates and returns an instance of ListSelectionHandler.
*/
protected ListSelectionListener createListSelectionListener ()
{return new ListSelectionHandler ();}

/**	If <code>updatingListSelectionModel</code> is false,
	reset the selected paths from the selected rows in the list
	selection model.
*/
protected void updateSelectedPathsFromSelectedRows ()
{
if (! updatingListSelectionModel)
	{
	updatingListSelectionModel = true;
	try
		{
		/*	This is way expensive, ListSelectionModel needs an
			enumerator for iterating.
		*/
		int
			min = listSelectionModel.getMinSelectionIndex (),
			max = listSelectionModel.getMaxSelectionIndex ();

		clearSelection ();
		if (min != -1 &&
			max != -1)
			{
			for (int counter = min;
					 counter <= max;
					 counter++)
				{
				if (listSelectionModel.isSelectedIndex (counter))
					{
					TreePath
						path = The_Tree.getPathForRow (counter);
					if (path != null)
						addSelectionPath (path);
					}
				}
			}
		}
	finally {updatingListSelectionModel = false;}
	}
}

/*------------------------------------------------------------------------------
*/
/**	Class responsible for calling updateSelectedPathsFromSelectedRows
	when the selection of the list changes.
*/
class ListSelectionHandler
	implements ListSelectionListener
{
public void valueChanged
	(
	ListSelectionEvent	event
	)
{updateSelectedPathsFromSelectedRows ();}

}	//	class ListSelectionHandler.

}	//	class ListToTreeSelectionModelWrapper.

}	//	class JTreeTable.
