% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/shinywrappers.R
\name{renderImage}
\alias{renderImage}
\title{Image file output}
\usage{
renderImage(
  expr,
  env = parent.frame(),
  quoted = FALSE,
  deleteFile,
  outputArgs = list()
)
}
\arguments{
\item{expr}{An expression that returns a list.}

\item{env}{The environment in which to evaluate \code{expr}.}

\item{quoted}{Is \code{expr} a quoted expression (with \code{quote()})? This
is useful if you want to save an expression in a variable.}

\item{deleteFile}{Should the file in \code{func()$src} be deleted after
it is sent to the client browser? Generally speaking, if the image is a
temp file generated within \code{func}, then this should be \code{TRUE};
if the image is not a temp file, this should be \code{FALSE}. (For backward
compatibility reasons, if this argument is missing, a warning will be
emitted, and if the file is in the temp directory it will be deleted. In
the future, this warning will become an error.)}

\item{outputArgs}{A list of arguments to be passed through to the implicit
call to \code{\link[=imageOutput]{imageOutput()}} when \code{renderImage} is used in an
interactive R Markdown document.}
}
\description{
Renders a reactive image that is suitable for assigning to an \code{output}
slot.
}
\details{
The expression \code{expr} must return a list containing the attributes for
the \code{img} object on the client web page. For the image to display,
properly, the list must have at least one entry, \code{src}, which is the
path to the image file. It may also useful to have a \code{contentType}
entry specifying the MIME type of the image. If one is not provided,
\code{renderImage} will try to autodetect the type, based on the file
extension.

Other elements such as \code{width}, \code{height}, \code{class}, and
\code{alt}, can also be added to the list, and they will be used as
attributes in the \code{img} object.

The corresponding HTML output tag should be \code{div} or \code{img} and have
the CSS class name \code{shiny-image-output}.
}
\examples{
## Only run examples in interactive R sessions
if (interactive()) {
options(device.ask.default = FALSE)

ui <- fluidPage(
  sliderInput("n", "Number of observations", 2, 1000, 500),
  plotOutput("plot1"),
  plotOutput("plot2"),
  plotOutput("plot3")
)

server <- function(input, output, session) {

  # A plot of fixed size
  output$plot1 <- renderImage({
    # A temp file to save the output. It will be deleted after renderImage
    # sends it, because deleteFile=TRUE.
    outfile <- tempfile(fileext='.png')

    # Generate a png
    png(outfile, width=400, height=400)
    hist(rnorm(input$n))
    dev.off()

    # Return a list
    list(src = outfile,
         alt = "This is alternate text")
  }, deleteFile = TRUE)

  # A dynamically-sized plot
  output$plot2 <- renderImage({
    # Read plot2's width and height. These are reactive values, so this
    # expression will re-run whenever these values change.
    width  <- session$clientData$output_plot2_width
    height <- session$clientData$output_plot2_height

    # A temp file to save the output.
    outfile <- tempfile(fileext='.png')

    png(outfile, width=width, height=height)
    hist(rnorm(input$n))
    dev.off()

    # Return a list containing the filename
    list(src = outfile,
         width = width,
         height = height,
         alt = "This is alternate text")
  }, deleteFile = TRUE)

  # Send a pre-rendered image, and don't delete the image after sending it
  # NOTE: For this example to work, it would require files in a subdirectory
  # named images/
  output$plot3 <- renderImage({
    # When input$n is 1, filename is ./images/image1.jpeg
    filename <- normalizePath(file.path('./images',
                              paste('image', input$n, '.jpeg', sep='')))

    # Return a list containing the filename
    list(src = filename)
  }, deleteFile = FALSE)
}

shinyApp(ui, server)
}
}
\seealso{
For more details on how the images are generated, and how to control
the output, see \code{\link[=plotPNG]{plotPNG()}}.
}
