//! Common definitions for the generate-tests tool.
/*
 * Copyright (c) 2022  Peter Pentchev <roam@ringlet.net>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

use std::path::PathBuf;

use config_diag::ConfigDiag;
use serde::{Deserialize, Serialize};

/// Runtime configuration for the `generate_tests` tool.
#[derive(Debug)]
pub struct Config {
    /// The path to the shell program to generate.
    pub output_file: PathBuf,

    /// Verbose operation: display diagnostic messages.
    pub verbose: bool,
}

impl ConfigDiag for Config {
    fn diag_is_verbose(&self) -> bool {
        self.verbose
    }
}

/// The description of a single test case.
#[derive(Debug, Deserialize, Serialize)]
pub struct TestCase {
    /// The human-readable name of the test case.
    pub title: String,

    /// Is this test supposed to succeed?
    pub success: bool,

    /// The name of the file to send/execute.
    pub filename: String,

    /// The regular expression to match the remrun output against.
    pub output_template: String,

    /// The options to pass to the program to execute.
    pub program_options: Vec<String>,

    /// The options to pass to remrun.
    pub remrun_options: Vec<String>,

    /// The name of the file to redirect the standard input from.
    pub stdin_filename: Option<String>,
}

/// The full test data.
#[derive(Debug, Serialize)]
pub struct TestData {
    /// The test cases themselves.
    pub tests: Vec<TestCase>,
}
