#!vtk
#
# This file is part of Rheolef.
#
# Copyright (C) 2000-2009 Pierre Saramito 
#
# Rheolef is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# Rheolef is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with Rheolef; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# -------------------------------------------------------------------------
#
# elevation render for scalar data on unstructured 2d meshes
# in polydata format
#
# date: 2 may 2001
#
# author: Pierre.Saramito@imag.fr
#
# NOTE: tested on vtk-3.2
#
# ----------------------------------------------------------
# usage
# ----------------------------------------------------------
#
# for inlusion in a 'wrapper.tcl' file:
#
#       set file_name "my-file.vtk"
#	source vtk_interactor.tcl
#	source vtk_elevation.tcl
# then
#       vtk wrapper.tcl
#
# -----------------------------------------------------
# parameters
# -----------------------------------------------------

set axis_color "0. 0. 0."

# -----------------------------------------------------
# render window
# -----------------------------------------------------

vtkRenderer render
    render SetBackground 1 1 1

# -----------------------------------------------------
# reader and bounds
# -----------------------------------------------------

vtkPolyDataReader reader
    reader SetFileName $file_name
    reader Update

set  bounds [[reader GetOutput] GetBounds]
scan $bounds "%f %f %f %f %f %f" minx maxx miny maxy minz maxz

set xmid [expr ($maxx + $minx)/2.0];
set ymid [expr ($maxy + $miny)/2.0];
set zmid [expr ($maxz + $minz)/2.0];
set xdiff [expr $maxx - $minx];
set ydiff [expr $maxy - $miny];
set zdiff [expr $maxz - $minz];

# -----------------------------------------------------
# geographic color table
# -----------------------------------------------------

vtkLookupTable color_table
    color_table SetHueRange        0.4 0.0
    color_table SetSaturationRange 1.0 0.1
    color_table SetValueRange      0.5 1.0

# -----------------------------------------------------
# elevation
# -----------------------------------------------------

vtkElevationFilter elevation
    elevation SetInput [reader GetOutput]
    eval elevation SetLowPoint 0 0 $minz
    eval elevation SetHighPoint 0 0 $maxz
    eval elevation SetScalarRange $minz $maxz

vtkDataSetMapper elevation_mapper
    elevation_mapper SetInput [elevation GetOutput]
    eval elevation_mapper SetScalarRange $minz $maxz
    elevation_mapper SetLookupTable color_table
    elevation_mapper ImmediateModeRenderingOn

vtkActor elevation_actor
    elevation_actor SetMapper elevation_mapper

render AddActor elevation_actor

# -----------------------------------------------------
# outline edges
# -----------------------------------------------------

vtkCubeSource outline
    outline SetXLength $xdiff;
    outline SetYLength $ydiff;
    outline SetZLength $zdiff;
    outline SetCenter  [ expr $xmid ] \
                       [ expr $ymid ] \
                       [ expr $zmid ]

vtkExtractEdges outline_edge
    outline_edge SetInput [outline GetOutput]

vtkPolyDataMapper outline_mapper
    outline_mapper SetInput [outline_edge GetOutput]
    outline_mapper ImmediateModeRenderingOn

vtkActor outline_actor
    outline_actor SetMapper outline_mapper
    eval [outline_actor GetProperty] SetColor    $axis_color
    [outline_actor GetProperty] SetAmbient  1
    [outline_actor GetProperty] SetDiffuse  0
    [outline_actor GetProperty] SetSpecular 0
    [outline_actor GetProperty] SetRepresentationToWireframe

render AddActor  outline_actor

# -----------------------------------------------------
# axis
# -----------------------------------------------------

vtkCubeAxesActor2D axis_actor
      axis_actor SetInput [reader GetOutput]
      axis_actor SetCamera [render GetActiveCamera]
      axis_actor SetLabelFormat "%6.4g"
      if { [get_vtk_major_version] >= 4 && [get_vtk_minor_version] > 0} {
	  vtkTextProperty axis_actor_property
	  axis_actor_property ShadowOn
	  axis_actor SetAxisLabelTextProperty axis_actor_property
	  axis_actor SetAxisTitleTextProperty axis_actor_property
      } else {
	  axis_actor ShadowOn
      }
      axis_actor SetFlyModeToOuterEdges
      axis_actor SetFontFactor 0.8
      eval [axis_actor GetProperty] SetColor $axis_color

render AddProp axis_actor

# -----------------------------------------------------
# color bar
# -----------------------------------------------------

vtkScalarBarActor bar_actor
    bar_actor SetLookupTable [elevation_mapper GetLookupTable]
    bar_actor SetOrientationToHorizontal
    [bar_actor GetProperty] SetColor 0 0 0
    [bar_actor GetPositionCoordinate] SetCoordinateSystemToNormalizedViewport
    [bar_actor GetPositionCoordinate] SetValue 0.1 0.01
    bar_actor SetOrientationToHorizontal
    bar_actor SetWidth  0.8
    bar_actor SetHeight 0.10
    bar_actor SetTitle "altitude"
    if { [get_vtk_major_version] >= 4 && [get_vtk_minor_version] > 0} {
	vtkTextProperty bar_text_property
        bar_text_property BoldOn
	bar SetTitleTextProperty bar_text_property
	bar SetLabelTextProperty bar_text_property
    } else {
        bar_actor BoldOn
    }
    eval [bar_actor GetProperty] SetColor $axis_color

render AddActor bar_actor

# -----------------------------------------------------
# action
# -----------------------------------------------------

vtkRenderWindow render_window
    render_window AddRenderer render
    render_window SetSize 500 500

vtkRenderWindowInteractor interactor
    interactor SetRenderWindow render_window
    if { [get_vtk_major_version] >= 4 && [get_vtk_minor_version] > 0} {
        iren AddObserver UserEvent {wm deiconify .vtkInteract};
    } else {
        iren SetUserMethod {wm deiconify .vtkInteract};
    }
    interactor Initialize

# position of the camera
[render GetActiveCamera] Zoom 1.4
[render GetActiveCamera] Roll -45	;# Rotate in xy plane
[render GetActiveCamera] Elevation 91	;# Rotate also

render_window Render

wm withdraw .


