#ifndef __GDK_EVENTS_H__
#define __GDK_EVENTS_H__

#include <gdk/gdktypes.h>
#include <gdk/gdkdnd.h>
#include <gdk/gdkinput.h>

#ifdef __cplusplus
extern "C" {
#endif                          /* __cplusplus */

#define	  GDK_PRIORITY_EVENTS		(G_PRIORITY_DEFAULT)

   typedef struct _GdkEventAny GdkEventAny;
   typedef struct _GdkEventExpose GdkEventExpose;
   typedef struct _GdkEventNoExpose GdkEventNoExpose;
   typedef struct _GdkEventVisibility GdkEventVisibility;
   typedef struct _GdkEventMotion GdkEventMotion;
   typedef struct _GdkEventButton GdkEventButton;
   typedef struct _GdkEventScroll GdkEventScroll;
   typedef struct _GdkEventKey GdkEventKey;
   typedef struct _GdkEventFocus GdkEventFocus;
   typedef struct _GdkEventCrossing GdkEventCrossing;
   typedef struct _GdkEventConfigure GdkEventConfigure;
   typedef struct _GdkEventProperty GdkEventProperty;
   typedef struct _GdkEventSelection GdkEventSelection;
   typedef struct _GdkEventProximity GdkEventProximity;
   typedef struct _GdkEventClient GdkEventClient;

   typedef struct _GdkEventDND GdkEventDND;

   typedef union _GdkEvent GdkEvent;

   typedef void (*GdkEventFunc) (GdkEvent * event, gpointer data);

/* Event filtering */

   typedef void GdkXEvent;      /* Can be cast to XEvent */

   typedef enum {
      GDK_FILTER_CONTINUE,      /* Event not handled, continue processesing */
      GDK_FILTER_TRANSLATE,     /* Translated event stored */
      GDK_FILTER_REMOVE         /* Terminate processing, removing event */
   } GdkFilterReturn;

   typedef GdkFilterReturn(*GdkFilterFunc) (GdkXEvent * xevent,
                                            GdkEvent * event,
                                            gpointer data);


/* Event types.
 *   Nothing: No event occurred.
 *   Delete: A window delete event was sent by the window manager.
 *	     The specified window should be deleted.
 *   Destroy: A window has been destroyed.
 *   Expose: Part of a window has been uncovered.
 *   NoExpose: Same as expose, but no expose event was generated.
 *   VisibilityNotify: A window has become fully/partially/not obscured.
 *   MotionNotify: The mouse has moved.
 *   ButtonPress: A mouse button was pressed.
 *   ButtonRelease: A mouse button was release.
 *   KeyPress: A key was pressed.
 *   KeyRelease: A key was released.
 *   EnterNotify: A window was entered.
 *   LeaveNotify: A window was exited.
 *   FocusChange: The focus window has changed. (The focus window gets
 *		  keyboard events).
 *   Resize: A window has been resized.
 *   Map: A window has been mapped. (It is now visible on the screen).
 *   Unmap: A window has been unmapped. (It is no longer visible on
 *	    the screen).
 *   Scroll: A mouse wheel was scrolled either up or down.
 */
   typedef enum {
      GDK_NOTHING = -1,
      GDK_DELETE = 0,
      GDK_DESTROY = 1,
      GDK_EXPOSE = 2,
      GDK_MOTION_NOTIFY = 3,
      GDK_BUTTON_PRESS = 4,
      GDK_2BUTTON_PRESS = 5,
      GDK_3BUTTON_PRESS = 6,
      GDK_BUTTON_RELEASE = 7,
      GDK_KEY_PRESS = 8,
      GDK_KEY_RELEASE = 9,
      GDK_ENTER_NOTIFY = 10,
      GDK_LEAVE_NOTIFY = 11,
      GDK_FOCUS_CHANGE = 12,
      GDK_CONFIGURE = 13,
      GDK_MAP = 14,
      GDK_UNMAP = 15,
      GDK_PROPERTY_NOTIFY = 16,
      GDK_SELECTION_CLEAR = 17,
      GDK_SELECTION_REQUEST = 18,
      GDK_SELECTION_NOTIFY = 19,
      GDK_PROXIMITY_IN = 20,
      GDK_PROXIMITY_OUT = 21,
      GDK_DRAG_ENTER = 22,
      GDK_DRAG_LEAVE = 23,
      GDK_DRAG_MOTION = 24,
      GDK_DRAG_STATUS = 25,
      GDK_DROP_START = 26,
      GDK_DROP_FINISHED = 27,
      GDK_CLIENT_EVENT = 28,
      GDK_VISIBILITY_NOTIFY = 29,
      GDK_NO_EXPOSE = 30,
      GDK_SCROLL = 31
   } GdkEventType;

/* Event masks. (Used to select what types of events a window
 *  will receive).
 */
   typedef enum {
      GDK_EXPOSURE_MASK = 1 << 1,
      GDK_POINTER_MOTION_MASK = 1 << 2,
      GDK_POINTER_MOTION_HINT_MASK = 1 << 3,
      GDK_BUTTON_MOTION_MASK = 1 << 4,
      GDK_BUTTON1_MOTION_MASK = 1 << 5,
      GDK_BUTTON2_MOTION_MASK = 1 << 6,
      GDK_BUTTON3_MOTION_MASK = 1 << 7,
      GDK_BUTTON_PRESS_MASK = 1 << 8,
      GDK_BUTTON_RELEASE_MASK = 1 << 9,
      GDK_KEY_PRESS_MASK = 1 << 10,
      GDK_KEY_RELEASE_MASK = 1 << 11,
      GDK_ENTER_NOTIFY_MASK = 1 << 12,
      GDK_LEAVE_NOTIFY_MASK = 1 << 13,
      GDK_FOCUS_CHANGE_MASK = 1 << 14,
      GDK_STRUCTURE_MASK = 1 << 15,
      GDK_PROPERTY_CHANGE_MASK = 1 << 16,
      GDK_VISIBILITY_NOTIFY_MASK = 1 << 17,
      GDK_PROXIMITY_IN_MASK = 1 << 18,
      GDK_PROXIMITY_OUT_MASK = 1 << 19,
      GDK_SUBSTRUCTURE_MASK = 1 << 20,
      GDK_SCROLL_MASK = 1 << 21,
      GDK_ALL_EVENTS_MASK = 0x3FFFFE
   } GdkEventMask;

   typedef enum {
      GDK_VISIBILITY_UNOBSCURED,
      GDK_VISIBILITY_PARTIAL,
      GDK_VISIBILITY_FULLY_OBSCURED
   } GdkVisibilityState;

   typedef enum {
      GDK_SCROLL_UP,
      GDK_SCROLL_DOWN,
      GDK_SCROLL_LEFT,
      GDK_SCROLL_RIGHT
   } GdkScrollDirection;

/* Types of enter/leave notifications.
 *   Ancestor:
 *   Virtual:
 *   Inferior:
 *   Nonlinear:
 *   NonlinearVirtual:
 *   Unknown: An unknown type of enter/leave event occurred.
 */
   typedef enum {
      GDK_NOTIFY_ANCESTOR = 0,
      GDK_NOTIFY_VIRTUAL = 1,
      GDK_NOTIFY_INFERIOR = 2,
      GDK_NOTIFY_NONLINEAR = 3,
      GDK_NOTIFY_NONLINEAR_VIRTUAL = 4,
      GDK_NOTIFY_UNKNOWN = 5
   } GdkNotifyType;

/* Enter/leave event modes.
 *   NotifyNormal
 *   NotifyGrab
 *   NotifyUngrab
 */
   typedef enum {
      GDK_CROSSING_NORMAL,
      GDK_CROSSING_GRAB,
      GDK_CROSSING_UNGRAB
   } GdkCrossingMode;

   typedef enum {
      GDK_PROPERTY_NEW_VALUE,
      GDK_PROPERTY_DELETE
   } GdkPropertyState;

   struct _GdkEventAny {
      GdkEventType type;
      GdkWindow *window;
      gint8 send_event;
   };

   struct _GdkEventExpose {
      GdkEventType type;
      GdkWindow *window;
      gint8 send_event;
      GdkRectangle area;
      gint count;               /* If non-zero, how many more events follow. */
   };

   struct _GdkEventNoExpose {
      GdkEventType type;
      GdkWindow *window;
      gint8 send_event;
      /* XXX: does anyone need the X major_code or minor_code fields? */
   };

   struct _GdkEventVisibility {
      GdkEventType type;
      GdkWindow *window;
      gint8 send_event;
      GdkVisibilityState state;
   };

   struct _GdkEventMotion {
      GdkEventType type;
      GdkWindow *window;
      gint8 send_event;
      guint32 time;
      gdouble x;
      gdouble y;
      gdouble pressure;
      gdouble xtilt;
      gdouble ytilt;
      guint state;
      gint16 is_hint;
      GdkInputSource source;
      guint32 deviceid;
      gdouble x_root, y_root;
   };

   struct _GdkEventButton {
      GdkEventType type;
      GdkWindow *window;
      gint8 send_event;
      guint32 time;
      gdouble x;
      gdouble y;
      gdouble pressure;
      gdouble xtilt;
      gdouble ytilt;
      guint state;
      guint button;
      GdkInputSource source;
      guint32 deviceid;
      gdouble x_root, y_root;
   };

   struct _GdkEventScroll {
      GdkEventType type;
      GdkWindow *window;
      gint8 send_event;
      guint32 time;
      gdouble x;
      gdouble y;
      gdouble pressure;
      gdouble xtilt;
      gdouble ytilt;
      guint state;
      GdkScrollDirection direction;
      GdkInputSource source;
      guint32 deviceid;
      gdouble x_root, y_root;
   };

   struct _GdkEventKey {
      GdkEventType type;
      GdkWindow *window;
      gint8 send_event;
      guint32 time;
      guint state;
      guint keyval;
      gint length;
      gchar *string;
   };

   struct _GdkEventCrossing {
      GdkEventType type;
      GdkWindow *window;
      gint8 send_event;
      GdkWindow *subwindow;
      guint32 time;
      gdouble x;
      gdouble y;
      gdouble x_root;
      gdouble y_root;
      GdkCrossingMode mode;
      GdkNotifyType detail;
      gboolean focus;
      guint state;
   };

   struct _GdkEventFocus {
      GdkEventType type;
      GdkWindow *window;
      gint8 send_event;
      gint16 in;
   };

   struct _GdkEventConfigure {
      GdkEventType type;
      GdkWindow *window;
      gint8 send_event;
      gint16 x, y;
      gint16 width;
      gint16 height;
   };

   struct _GdkEventProperty {
      GdkEventType type;
      GdkWindow *window;
      gint8 send_event;
      GdkAtom atom;
      guint32 time;
      guint state;
   };

   struct _GdkEventSelection {
      GdkEventType type;
      GdkWindow *window;
      gint8 send_event;
      GdkAtom selection;
      GdkAtom target;
      GdkAtom property;
      guint32 requestor;
      guint32 time;
   };

/* This event type will be used pretty rarely. It only is important
   for XInput aware programs that are drawing their own cursor */

   struct _GdkEventProximity {
      GdkEventType type;
      GdkWindow *window;
      gint8 send_event;
      guint32 time;
      GdkInputSource source;
      guint32 deviceid;
   };

   struct _GdkEventClient {
      GdkEventType type;
      GdkWindow *window;
      gint8 send_event;
      GdkAtom message_type;
      gushort data_format;
      union {
         char b[20];
         short s[10];
         long l[5];
      } data;
   };

/* Event types for DND */

   struct _GdkEventDND {
      GdkEventType type;
      GdkWindow *window;
      gint8 send_event;
      GdkDragContext *context;

      guint32 time;
      gshort x_root, y_root;
   };

   union _GdkEvent {
      GdkEventType type;
      GdkEventAny any;
      GdkEventExpose expose;
      GdkEventNoExpose no_expose;
      GdkEventVisibility visibility;
      GdkEventMotion motion;
      GdkEventButton button;
      GdkEventScroll scroll;
      GdkEventKey key;
      GdkEventCrossing crossing;
      GdkEventFocus focus_change;
      GdkEventConfigure configure;
      GdkEventProperty property;
      GdkEventSelection selection;
      GdkEventProximity proximity;
      GdkEventClient client;
      GdkEventDND dnd;
   };

   gboolean gdk_events_pending(void);
   GdkEvent *gdk_event_get(void);

   GdkEvent *gdk_event_peek(void);
   GdkEvent *gdk_event_get_graphics_expose(GdkWindow * window);
   void gdk_event_put(GdkEvent * event);

   GdkEvent *gdk_event_copy(GdkEvent * event);
   void gdk_event_free(GdkEvent * event);
   guint32 gdk_event_get_time(GdkEvent * event);

   void gdk_event_handler_set(GdkEventFunc func,
                              gpointer data, GDestroyNotify notify);

   void gdk_set_show_events(gboolean show_events);
   gboolean gdk_get_show_events(void);

/*
 * The following function adds a global filter for all client
 * messages of type message_type
 */
   void gdk_add_client_message_filter(GdkAtom message_type,
                                      GdkFilterFunc func, gpointer data);

   gboolean gdk_check_typed_window_event(GdkWindow * w, gint type,
                                         GdkEvent * event);

#ifdef __cplusplus
}
#endif                          /* __cplusplus */
#endif                          /* __GDK_EVENTS_H__ */
