/*
*  RAL -- Rubrica Addressbook Library
*  file: ref.c
*  
*  Copyright (C) Nicola Fragale <nicolafragale@libero.it>
*
*  This program is free software; you can redistribute it and/or modify
*  it under the terms of the GNU General Public License as published by
*  the Free Software Foundation; either version 2 of the License, or
*  (at your option) any later version.
*
*  This program is distributed in the hope that it will be useful,
*  but WITHOUT ANY WARRANTY; without even the implied warranty of
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*  GNU General Public License for more details.
*
*  You should have received a copy of the GNU General Public License
*  along with this program; if not, write to the Free Software
*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include <gtk/gtk.h>
#include <glib/gi18n-lib.h>
#include <gconf/gconf-client.h>
#include <errno.h>

#include "libral.h"

#include "cards_view.h"
#include "preferences.h"



enum {
  PROP_0,
  VIEW_HAS_CHANGED,
  VIEW_BROTHER,
};




static GObjectClass* parent_class = NULL;
static gint order[COLUMNS] = {0};


struct _RubricaCardsViewPrivate {
  RAbook*    abook;
  GtkWidget* brother;
  gboolean   change;

  gboolean dispose_has_run;
};



#define RUBRICA_CARDS_VIEW_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE((o),  \
                                           RUBRICA_CARDS_VIEW_TYPE,          \
                                           RubricaCardsViewPrivate))



static void rubrica_cards_view_class_init   (RubricaCardsViewClass* klass);
static void rubrica_cards_view_init         (RubricaCardsView* obj);

static void rubrica_cards_view_finalize     (RubricaCardsView* self);
static void rubrica_cards_view_dispose      (RubricaCardsView* self);


static void rubrica_cards_view_set_property (GObject* obj, guint property_id,
					     GValue* value,  GParamSpec* spec);
static void rubrica_cards_view_get_property (GObject* obj, guint property_id,
					     GValue* value, GParamSpec* spec);


/* Private 
 */
static void     cards_view_load_file    (RubricaCardsView* view, gchar* file);
static void     cards_update_infos      (RubricaCardsView* view);

static void     cards_view_add_card     (RubricaCardsView* view, RCard* card);
static gboolean cards_view_delete_card  (GtkTreeModel *model, 
					 GtkTreePath *path,
					 GtkTreeIter *iter, gpointer data);
static void     cards_view_show_column  (GConfClient *client, guint cnxn_id, 
					 GConfEntry *entry, gpointer data);
static void     cards_view_setup_column (GtkTreeViewColumn* column, 
					 gchar* key);
static void     cards_view_add_columns  (GtkTreeView *tree);

// Callbacks
static void     on_open_file_failed_cb  (RAbook* book, gint error, 
					 gpointer data);




static void 
cards_view_load_file(RubricaCardsView* view, gchar* filename)
{
  RubricaCardsViewPrivate* priv = RUBRICA_CARDS_VIEW_GET_PRIVATE(view);
  
  if (r_abook_open_file(R_ABOOK(priv->abook), filename))
    {
      RCard* card;

      r_abook_reset_book(priv->abook);
      card = r_abook_get_card(priv->abook);
      for (; card; card = r_abook_get_next_card(priv->abook))
	cards_view_add_card(view, card);
      
      cards_update_infos(view);
    }  
}


static void 
cards_update_infos(RubricaCardsView* view)
{
  gint items, deleted;
  gchar* message;
  
  items   = r_abook_get_items(view->priv->abook);
  deleted = r_abook_get_deleted(view->priv->abook);
  message = g_strdup_printf(_("Cards in this addressbook: %-5d "
			      "Deleted cards: %-5d"), items, deleted);
  g_object_set(view, "view-message", message, NULL);
  
  g_signal_emit_by_name(view, "view-changed", message, G_TYPE_STRING);
}


static void
cards_view_add_card(RubricaCardsView* view, RCard* card)
{
  GtkWidget    *tree;
  GtkTreeModel *model;
  GtkTreeIter  iter;
  RInfos *infos;
    
  g_return_if_fail(IS_RUBRICA_CARDS_VIEW(view));
  g_return_if_fail(IS_R_CARD(card));
  
  infos = r_card_get_infos(R_CARD(card));
  tree  = rubrica_view_get_tree(RUBRICA_VIEW(view));
  model = gtk_tree_view_get_model(GTK_TREE_VIEW(tree));    
  
  gtk_list_store_append(GTK_LIST_STORE(model), &iter);  
  if (r_card_is_personal(card))
    {
      gtk_list_store_set(GTK_LIST_STORE(model), &iter,
			 CARD_COLUMN,       r_infos_get_label(infos), 
			 FIRST_NAME_COLUMN, r_infos_get_first(infos), 
			 LAST_NAME_COLUMN,  r_infos_get_last(infos),
			 PROFESSION_COLUMN, r_infos_get_profession(infos),
			 CITY_COLUMN,       r_infos_get_city(infos),
			 COUNTRY_COLUMN,    r_infos_get_country(infos),
			 WEB_CARD_COLUMN,   r_infos_get_web(infos),
			 EMAIL_CARD_COLUMN, r_infos_get_email(infos),
			 TELEPHONE_COLUMN,  r_infos_get_telephone(infos),
			 ASSIGNMENT_COLUMN, r_infos_get_assignment(infos),
			 ID_COLUMN,         r_infos_get_id(infos),
			 -1);
    }
  else
    {
      gtk_list_store_set(GTK_LIST_STORE(model), &iter,
			 CARD_COLUMN,       r_infos_get_label(infos), 
			 CITY_COLUMN,       r_infos_get_city(infos),
			 COUNTRY_COLUMN,    r_infos_get_country(infos),
			 WEB_CARD_COLUMN,   r_infos_get_web(infos),
			 EMAIL_CARD_COLUMN, r_infos_get_email(infos),
			 TELEPHONE_COLUMN,  r_infos_get_telephone(infos),
			 ID_COLUMN,         r_infos_get_id(infos),
			 -1); 
    }

  r_card_free_infos(infos);
}


static gboolean
cards_view_delete_card  (GtkTreeModel *model, GtkTreePath *path,
			 GtkTreeIter *iter, gpointer data)
{
  gulong card_id = (gulong) data;
  gulong id;
  
  gtk_tree_model_get(model, iter, ID_COLUMN, &id, -1);  
  if (id == card_id)
    {
      gtk_list_store_remove(GTK_LIST_STORE(model), iter);  
      
      return TRUE;
    }

  return FALSE;
}


static void 
cards_view_show_column(GConfClient *client, guint cnxn_id, 
		       GConfEntry *entry, gpointer data)
{
  gboolean bool;  
  GtkTreeViewColumn* column = (GtkTreeViewColumn*) data;
  
  bool = gconf_value_get_bool (entry->value);
  gtk_tree_view_column_set_visible(column, bool);
}


static void
cards_view_setup_column (GtkTreeViewColumn* column, gchar* key)
{
  GConfClient* client;
  gboolean bool;
  
  client = gconf_client_get_default();

  bool = gconf_client_get_bool(client, key, NULL);
  gtk_tree_view_column_set_visible(column, bool);

  gconf_client_notify_add(client, key, cards_view_show_column, column,
			  NULL, NULL);
  
  g_object_unref(client);  
}


static void 
cards_view_add_columns (GtkTreeView *tree)
{
  GtkTreeViewColumn* column;
  GtkCellRenderer *renderer;
  GtkTreeModel *model;
  gint col;

  model = gtk_tree_view_get_model(GTK_TREE_VIEW(tree));
  
  renderer = gtk_cell_renderer_text_new();
  col = gtk_tree_view_insert_column_with_attributes(tree, -1, _("Card"), 
						    renderer, "text", 
						    CARD_COLUMN, NULL);
  order[CARD_COLUMN] = col;
  column = gtk_tree_view_get_column(tree, col-1);
  gtk_tree_view_column_set_reorderable(GTK_TREE_VIEW_COLUMN(column), TRUE);
  gtk_tree_view_column_set_sort_column_id(GTK_TREE_VIEW_COLUMN(column), 
					  CARD_COLUMN);
  cards_view_setup_column(column, RUBRICA_GCONF_CARD);
  
  renderer = gtk_cell_renderer_text_new();
  col = gtk_tree_view_insert_column_with_attributes(tree, -1, 
						    _("First Name"),
						    renderer, "text", 
						    FIRST_NAME_COLUMN,
						    NULL);
  order[FIRST_NAME_COLUMN] = col;
  column = gtk_tree_view_get_column(tree, col-1);
  gtk_tree_view_column_set_reorderable(GTK_TREE_VIEW_COLUMN(column), TRUE);
  gtk_tree_view_column_set_sort_column_id(GTK_TREE_VIEW_COLUMN(column), 
					  FIRST_NAME_COLUMN);
  cards_view_setup_column(column, RUBRICA_GCONF_FIRST_NAME);
  
  renderer = gtk_cell_renderer_text_new();
  col = gtk_tree_view_insert_column_with_attributes (tree, -1, 
						     _("Last Name"),
						     renderer, "text", 
						     LAST_NAME_COLUMN,
						     NULL);
  order[LAST_NAME_COLUMN] = col;
  column = gtk_tree_view_get_column(tree, col-1);
  gtk_tree_view_column_set_reorderable(GTK_TREE_VIEW_COLUMN(column), TRUE);
  gtk_tree_view_column_set_sort_column_id(GTK_TREE_VIEW_COLUMN(column), 
					  LAST_NAME_COLUMN);
  cards_view_setup_column(column, RUBRICA_GCONF_LAST_NAME);
  
  renderer = gtk_cell_renderer_text_new();
  col = gtk_tree_view_insert_column_with_attributes (tree, -1,
						     _("Profession"),
						     renderer,  "text",
						     PROFESSION_COLUMN, 
						     NULL);
  order[PROFESSION_COLUMN] = col;
  column = gtk_tree_view_get_column(tree, col-1);
  gtk_tree_view_column_set_reorderable(GTK_TREE_VIEW_COLUMN(column), TRUE);
  gtk_tree_view_column_set_sort_column_id(GTK_TREE_VIEW_COLUMN(column), 
					  PROFESSION_COLUMN);
  cards_view_setup_column(column, RUBRICA_GCONF_PROFESSION);
  
  renderer = gtk_cell_renderer_text_new();
  col = gtk_tree_view_insert_column_with_attributes (tree, -1, _("City"), 
						     renderer, "text", 
						     CITY_COLUMN, NULL);
  order[CITY_COLUMN] = col;
  column = gtk_tree_view_get_column(tree, col-1);
  gtk_tree_view_column_set_reorderable(GTK_TREE_VIEW_COLUMN(column), TRUE);
  gtk_tree_view_column_set_sort_column_id(GTK_TREE_VIEW_COLUMN(column), 
					  CITY_COLUMN);
  cards_view_setup_column(column, RUBRICA_GCONF_CITY);
  
  renderer = gtk_cell_renderer_text_new();
  col = gtk_tree_view_insert_column_with_attributes (tree, -1, 
						     _("Country"), 
						     renderer, "text", 
						     COUNTRY_COLUMN,
						     NULL);
  order[COUNTRY_COLUMN] = col;
  column = gtk_tree_view_get_column(tree, col-1);
  gtk_tree_view_column_set_reorderable(GTK_TREE_VIEW_COLUMN(column), TRUE);
  gtk_tree_view_column_set_sort_column_id(GTK_TREE_VIEW_COLUMN(column),
					  COUNTRY_COLUMN);
  cards_view_setup_column(column, RUBRICA_GCONF_COUNTRY);
  
  renderer = gtk_cell_renderer_text_new();
  col = gtk_tree_view_insert_column_with_attributes (tree, -1, _("Web"),
						     renderer, "text",
						     WEB_CARD_COLUMN,
						     NULL);
  order[WEB_CARD_COLUMN] = col;
  column = gtk_tree_view_get_column(tree, col-1);
  gtk_tree_view_column_set_reorderable(GTK_TREE_VIEW_COLUMN(column), TRUE);
  gtk_tree_view_column_set_sort_column_id(GTK_TREE_VIEW_COLUMN(column), 
					  WEB_CARD_COLUMN);
  cards_view_setup_column(column, RUBRICA_GCONF_WEB);
  
  renderer = gtk_cell_renderer_text_new();
  col = gtk_tree_view_insert_column_with_attributes (tree, -1, _("Email"),
						     renderer, "text",
						     EMAIL_CARD_COLUMN, 
						     NULL);
  order[EMAIL_CARD_COLUMN] = col;
  column = gtk_tree_view_get_column(tree, col-1);
  gtk_tree_view_column_set_reorderable(GTK_TREE_VIEW_COLUMN(column), TRUE);
  gtk_tree_view_column_set_sort_column_id(GTK_TREE_VIEW_COLUMN(column), 
					  EMAIL_CARD_COLUMN);
  cards_view_setup_column(column, RUBRICA_GCONF_EMAIL);
  
  renderer = gtk_cell_renderer_text_new();
  col = gtk_tree_view_insert_column_with_attributes (tree, -1, 
						     _("Telephone"),
						     renderer, "text", 
						     TELEPHONE_COLUMN,
						     NULL);
  order[TELEPHONE_COLUMN] = col;
  column = gtk_tree_view_get_column(tree, col-1);
  gtk_tree_view_column_set_reorderable(GTK_TREE_VIEW_COLUMN(column), TRUE);
  gtk_tree_view_column_set_sort_column_id(GTK_TREE_VIEW_COLUMN(column),
					  TELEPHONE_COLUMN);
  cards_view_setup_column(column, RUBRICA_GCONF_TELEPHONE);
  
  renderer = gtk_cell_renderer_text_new();
  col = gtk_tree_view_insert_column_with_attributes (tree, -1, 
						     _("Company"), 
						     renderer, "text", 
						     COMPANY_COLUMN, NULL);
  order[COMPANY_COLUMN] = col;
  column = gtk_tree_view_get_column(tree, col-1);
  gtk_tree_view_column_set_reorderable(GTK_TREE_VIEW_COLUMN(column), TRUE);
  gtk_tree_view_column_set_sort_column_id(GTK_TREE_VIEW_COLUMN(column),
					  COMPANY_COLUMN);
  cards_view_setup_column(column, RUBRICA_GCONF_COMPANY);
  
  renderer = gtk_cell_renderer_text_new();
  col = gtk_tree_view_insert_column_with_attributes (tree, -1, 
						     _("Assignment"),
						     renderer, "text",
						     ASSIGNMENT_COLUMN,
						     NULL);
  order[ASSIGNMENT_COLUMN] = col;
  column = gtk_tree_view_get_column(tree, col-1);     
  gtk_tree_view_column_set_reorderable(GTK_TREE_VIEW_COLUMN(column), TRUE);
  gtk_tree_view_column_set_sort_column_id(GTK_TREE_VIEW_COLUMN(column),
					  ASSIGNMENT_COLUMN);
  cards_view_setup_column(column, RUBRICA_GCONF_ASSIGNMENT);      
  
  gtk_tree_sortable_set_sort_column_id(GTK_TREE_SORTABLE(model), 
				       CARD_COLUMN, GTK_SORT_ASCENDING);  
}


/*    Callbacks
*/
static void 
on_open_file_failed_cb(RAbook* book, gint error, gpointer data)
{
  GtkWidget* dialog;
  gchar* file;
  
  g_object_get(book, "addressbook-name", &file, NULL);
  
  dialog = gtk_message_dialog_new (NULL,
				   GTK_DIALOG_DESTROY_WITH_PARENT,
				   GTK_MESSAGE_ERROR,
				   GTK_BUTTONS_CLOSE,
				   _("Error loading file '%s': %s"),
				   file, g_strerror (errno));

  gtk_dialog_run (GTK_DIALOG (dialog));
  gtk_widget_destroy (dialog);  
}




/*  Public
*/
GType
rubrica_cards_view_get_type()
{
  static GType cards_view_type = 0;
  
  if (!cards_view_type)
    {
      static const GTypeInfo cards_view_info =
	{
	  sizeof(RubricaCardsViewClass),
	  NULL,
	  NULL,
	  (GClassInitFunc) rubrica_cards_view_class_init,
	  NULL,
	  NULL,
	  sizeof(RubricaCardsView),
	  0,
	  (GInstanceInitFunc) rubrica_cards_view_init
	};

      cards_view_type = g_type_register_static (RUBRICA_VIEW_TYPE, 
						"RubricaCardsView", 
						&cards_view_info, 0);
    }
  
  return cards_view_type;
}


static void 
rubrica_cards_view_dispose (RubricaCardsView* self)
{
  g_return_if_fail(IS_RUBRICA_CARDS_VIEW(self));
  
  if (self->priv->dispose_has_run)
    return;

  self->priv->dispose_has_run = TRUE;  
}


static void 
rubrica_cards_view_finalize (RubricaCardsView* self)
{
  g_return_if_fail(IS_RUBRICA_CARDS_VIEW(self));
}


static void
rubrica_cards_view_class_init(RubricaCardsViewClass* klass)
{
  GObjectClass *object_class = G_OBJECT_CLASS (klass);
  GParamSpec* pspec;
  
  parent_class = g_type_class_peek_parent (klass);

  object_class->finalize = (GObjectFinalizeFunc) rubrica_cards_view_finalize;
  object_class->dispose  = (GObjectFinalizeFunc) rubrica_cards_view_dispose;

  object_class->set_property = (gpointer) rubrica_cards_view_set_property;
  object_class->get_property = (gpointer) rubrica_cards_view_get_property;

  g_type_class_add_private (klass, sizeof(RubricaCardsViewPrivate));


  /**
   * RubricaCardsView:change
   *
   * does the addressbook change?
   */
  pspec = g_param_spec_boolean("has-changed",
			       "has changed",
			       "does view changed?",
			       FALSE,
			       G_PARAM_CONSTRUCT | G_PARAM_READWRITE);
  g_object_class_install_property(object_class, VIEW_HAS_CHANGED, pspec);

  /**
   * RubricaCardsView:brother
   *
   * the object that replace the notebook's tab
   */
  pspec = g_param_spec_object("view-brother", 
			      "view brother", 
			      "the object that replace the notebook's tab",  
			      GTK_TYPE_WIDGET,
			      G_PARAM_READWRITE);
  g_object_class_install_property(object_class, VIEW_BROTHER, pspec);  
}


static void
rubrica_cards_view_init(RubricaCardsView* self)
{
  GtkWidget*    tree;
  GtkTreeModel* model;
  gchar* message;
  
  g_return_if_fail(IS_RUBRICA_CARDS_VIEW(self));
  
  self->priv = RUBRICA_CARDS_VIEW_GET_PRIVATE(self);
 
  self->priv->abook   = r_abook_new();
  self->priv->brother = NULL;
  self->priv->change  = FALSE;

  model = (GtkTreeModel*) gtk_list_store_new(COLUMNS, 
					     G_TYPE_STRING,  /* card label */
					     G_TYPE_STRING,  /* first name */ 
					     G_TYPE_STRING,  /* last name  */ 
					     G_TYPE_STRING,  /* profession */
					     G_TYPE_STRING,  /* city       */
					     G_TYPE_STRING,  /* country    */
					     G_TYPE_STRING,  /* web        */
					     G_TYPE_STRING,  /* email      */ 
					     G_TYPE_STRING,  /* telephone  */ 
					     G_TYPE_STRING,  /* company    */ 
					     G_TYPE_STRING,  /* assigment  */
					     G_TYPE_INT);    /* card's id  */

  rubrica_view_set_model(RUBRICA_VIEW(self), model);

  tree = rubrica_view_get_tree(RUBRICA_VIEW(self));
  cards_view_add_columns (GTK_TREE_VIEW(tree));

  message = g_strdup_printf(_("Cards in this addressbook: %-5d "
			      "Deleted cards: %-5d"), 0, 0);
  g_object_set(self, "view-message", message, NULL);

  g_signal_connect(self->priv->abook, "open_fail",
		   G_CALLBACK (on_open_file_failed_cb), self);

  g_object_set(self->priv->abook, "addressbook-name", _("no name"), NULL);  

  self->priv->dispose_has_run = FALSE;
}


static void 
rubrica_cards_view_set_property (GObject* obj, guint property_id,
				 GValue* value, GParamSpec* spec)
{
  RubricaCardsView* self = (RubricaCardsView*) obj;
  RubricaCardsViewPrivate* priv = RUBRICA_CARDS_VIEW_GET_PRIVATE(self);

  switch (property_id) 
    {
    case VIEW_HAS_CHANGED:
      priv->change = g_value_get_boolean(value);
      break;

    case VIEW_BROTHER:
      priv->brother = g_value_get_object(value);
      break;

    default: 
      break; 
    } 
} 
 

static void 
rubrica_cards_view_get_property (GObject* obj, guint property_id,
				 GValue* value, GParamSpec* spec)
{
  RubricaCardsView* self = (RubricaCardsView*) obj;
  RubricaCardsViewPrivate* priv = RUBRICA_CARDS_VIEW_GET_PRIVATE(self);

  switch (property_id) 
    {      
    case VIEW_HAS_CHANGED:
      g_value_set_boolean(value, priv->change);
      break;

    case VIEW_BROTHER:
      g_value_set_object(value, priv->brother);
     break;

    default:
      break;  
    }  
}




/*   Public
*/
/**
 * rubrica_cards_view_new
 *
 * create a #RubricaCardsView
 *
 * returns: a new #RubricaCardsView*
 */
GtkWidget*
rubrica_cards_view_new(gchar* file)
{
  GtkWidget* view;

  view = GTK_WIDGET(g_object_new(RUBRICA_CARDS_VIEW_TYPE, 
				 "child-type", "Addressbook", NULL));
 
  if (file)
    cards_view_load_file(RUBRICA_CARDS_VIEW(view), file);

  return view;
}


gchar*  
rubrica_cards_view_get_addressbook_name(RubricaCardsView* view)
{
  gchar* name = NULL;

  g_return_val_if_fail(IS_RUBRICA_CARDS_VIEW(view), NULL);
  
  g_object_get(view->priv->abook, "addressbook-name", &name, NULL);

  return name;
}


RAbook* 
rubrica_cards_view_get_addressbook(RubricaCardsView* view)
{
  if (!view)
    return NULL;

  g_return_val_if_fail(IS_RUBRICA_CARDS_VIEW(view), NULL);
  
  return (RAbook*) view->priv->abook;
}


void 
rubrica_cards_view_add_card (RubricaCardsView* view, RCard* card)
{

  g_return_if_fail(IS_RUBRICA_CARDS_VIEW(view));

  cards_view_add_card(view, card);
  cards_update_infos(view);
}


void 
rubrica_cards_view_delete_card(RubricaCardsView* view, RCard* card)
{
  GtkWidget    *tree;
  GtkTreeModel *model;
  GtkTreeIter   iter;  
  GtkTreeSelection *selection;

  g_return_if_fail(IS_RUBRICA_CARDS_VIEW(view));
  
  tree      = rubrica_view_get_tree(RUBRICA_VIEW(view));
  model     = gtk_tree_view_get_model(GTK_TREE_VIEW(tree)); 
  selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(tree));

  if (gtk_tree_selection_get_selected(selection, NULL, &iter))
    gtk_list_store_remove(GTK_LIST_STORE(model), &iter);  
  else
    {
      gulong id;
      
      g_object_get(card, "card-id", &id, NULL);
    
      gtk_tree_model_foreach(GTK_TREE_MODEL(model), cards_view_delete_card, 
			     GINT_TO_POINTER(id));
    }
  
  cards_update_infos(view);
}


void 
rubrica_cards_view_modify_card(RubricaCardsView* view, RCard* card)
{
  GtkWidget        *tree;
  GtkTreeModel     *model;
  GtkTreeSelection *selection;
  GtkTreeIter       iter;  

  g_return_if_fail(IS_RUBRICA_CARDS_VIEW(view));

  tree  = rubrica_view_get_tree(RUBRICA_VIEW(view));
  model = gtk_tree_view_get_model(GTK_TREE_VIEW(tree));
  selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(tree));
  
  if (gtk_tree_selection_get_selected(selection, NULL, &iter))
    {
      RInfos* infos;

      infos = r_card_get_infos(R_CARD(card));
      
      gtk_list_store_remove(GTK_LIST_STORE(model), &iter);
      gtk_list_store_append(GTK_LIST_STORE(model), &iter);
      gtk_list_store_set(GTK_LIST_STORE(model), &iter,
			 CARD_COLUMN,       r_infos_get_label(infos), 
			 FIRST_NAME_COLUMN, r_infos_get_first(infos), 
			 LAST_NAME_COLUMN,  r_infos_get_last(infos),
			 PROFESSION_COLUMN, r_infos_get_profession(infos),
			 CITY_COLUMN,       r_infos_get_city(infos),
			 COUNTRY_COLUMN,    r_infos_get_country(infos),
			 WEB_CARD_COLUMN,   r_infos_get_web(infos),
			 EMAIL_CARD_COLUMN, r_infos_get_email(infos),
			 TELEPHONE_COLUMN,  r_infos_get_telephone(infos),
			 ASSIGNMENT_COLUMN, r_infos_get_assignment(infos),
			 ID_COLUMN,         r_infos_get_id(infos),
			 -1);   
      
      r_card_free_infos(infos);
    }  

  cards_update_infos(view);
}


void
rubrica_cards_view_display_results (RubricaCardsView* view, GList* list)
{
  GtkWidget    *tree;
  GtkTreeModel *model;   

  g_return_if_fail(IS_RUBRICA_CARDS_VIEW(view));

  tree  = rubrica_view_get_tree(RUBRICA_VIEW(view));
  model = gtk_tree_view_get_model(GTK_TREE_VIEW(tree)); 

  gtk_list_store_clear(GTK_LIST_STORE(model));
  for (; list; list = list->next)
    {
      RCard* card;
      glong id = (glong) list->data;
      
      card = r_abook_get_card_by_id(view->priv->abook, id);
      cards_view_add_card(view, card);
    }  

  cards_update_infos(view);
}


void
rubrica_cards_view_update_infos (RubricaCardsView* view)
{
  g_return_if_fail(IS_RUBRICA_CARDS_VIEW(view));

  cards_update_infos(view);  
}
