------------------------------------------------------------------------------
--  Thin Ada95 binding to OCI (Oracle Call Interface)                    --
--  Copyright (C) 2006-2007 Dmitriy Anisimkov.                              --
--  License agreement and authors contact information are in file oci.ads   --
------------------------------------------------------------------------------

--  $Id: test_lob.adb,v 1.15 2008/01/16 07:47:41 vagul Exp $

--  Example of using dynamically allocated bind variables.

with Ada.Calendar;
with Ada.Command_Line;
with Ada.Strings.Fixed;
with Ada.Text_IO;

with OCI.Thick.DB;
with OCI.Thick.Containers;
with OCI.Thick.Connections;
with OCI.Thick.Lobs;

procedure Test_Lob is

   --  TODO: non-blocking support
   --  !!! Multy non-blocking request fail because of Oracle Bug No. 5711162

   use Ada.Text_IO;
   use OCI.Thick;
   use OCI.Thick.Connections;
   use OCI.Thick.Containers;
   use OCI.Thick.DB;
   use ASCII;

   function Connect_String return String is
   begin
      if Ada.Command_Line.Argument_Count >= 1 then
         return Ada.Command_Line.Argument (1);
      else
         return "scott/tiger";
      end if;
   end Connect_String;

   Connect : Connection := Logon (Connect_String);

   Stmt_Lob : Statement
      := Prepare
           (Connect,
            "declare" & LF
          & "   procedure Append (Item in Varchar2) is" & LF
          & "   begin" & LF
          & "      DBMS_Lob.WriteAppend (:Loc, Length (Item), Item);" & LF
          & "   end Append;" & LF

          & "   procedure Delay is" & LF
          & "      pragma Autonomous_Transaction;" & LF
          & "      Dummy Integer;" & LF
          & "      Resource_Busy exception;" & LF
          & "      pragma Exception_Init (Resource_Busy, -30006);" & LF
          & "   begin" & LF
          & "     select DeptNo into Dummy from Dept where DeptNo = 10" & LF
          & "     for update wait 1;" & LF
          & "     Delay;" & LF
          & "     commit;" & LF
          & "   exception" & LF
          & "      when Resource_Busy then rollback;" & LF
          & "   end Delay;" & LF

          & "begin" & LF
          & "  if :Loc is null then" & LF
          & "     DBMS_Lob.CreateTemporary (:Loc, Cache => True);" & LF
          & "  end if;" & LF
          & "  if DBMS_Lob.GetLength (:Loc) = 10 then" & LF
          & "     Delay; " & LF
          & "  end if;" & LF
          & "  Append (DBMS_Lob.GetLength (:Loc) || :Line);" & LF
          & "end;");

   Loc    : Lobs.Locator := Lobs.Create (Connect, Lobs.Char);
   Line   : Positive   := 1;
   Offset : Lobs.Count := 1;

   procedure In_Lob (Data : out Data_Holder; Position, Iteration : Positive) is
      use Ada.Strings.Fixed;
      Name : constant String := Bind_Name (Stmt_Lob, Position);
   begin
      if Name = "LOC" then
         Data := To_Data (Loc);
      elsif Name = "LINE" then
         Data := To_Data ((Line rem 15) * Integer'Image (Line) & LF);
         Line := Line + 1;
      else
         raise Constraint_Error with "Wrong bind variable name: " & Name;
      end if;
   end In_Lob;

   procedure Out_Lob (Data : in Data_Holder; Position, Iteration : in Positive)
   is
      Name : constant String := Bind_Name (Stmt_Lob, Position);
   begin
      Loc := Value (Data);
   end Out_Lob;

begin
   Bind (Stmt_Lob, Type_Char_Lob, "Loc");
   Bind (Stmt_Lob, Type_String,   "Line");

   --  !!! Multy non-blocking request fail because of Oracle Bug No. 5711162
   --  Set_Blocking (Stmt_Lob, False);

   loop
      Execute (Stmt_Lob, In_Lob'Access, Out_Lob'Access, Count => 35);
      exit when not Is_Executing (Stmt_Lob);
      Put ('.');
      delay 0.1;
   end loop;

   Set_Blocking (Stmt_Lob, True);

   loop
      declare
         use type Lobs.Count;
         Line : String (1 .. 256);
         Last : Natural;
      begin
         Lobs.Read (Loc, Offset, Line, Last);
         Put (Line (1 .. Last));
         exit when Last < Line'Last;
         Offset := Offset + Lobs.Count (Last);
      end;
   end loop;

end Test_Lob;
