-- test/test-tongue.langpack.lua
--
-- Lua I18N library 'Tongue' -- Tests for the language pack module
--
-- Copyright 2016 Daniel Silverstone <dsilvers@digital-scurf.org>
--
-- For Licence terms, see COPYING
--

-- Step one, start coverage

pcall(require, 'luacov')

local tongue = require 'tongue'

local testnames = {}

local real_assert = assert
local total_asserts = 0
local function assert(...)
   local retval = real_assert(...)
   total_asserts = total_asserts + 1
   return retval
end

local function add_test(suite, name, value)
   rawset(suite, name, value)
   testnames[#testnames+1] = name
end

local suite = setmetatable({}, {__newindex = add_test})

function suite.fallback_present()
   assert(tongue.langpack.fallback, "Fallback language pack not present")
end

function suite.create_present()
   assert(tongue.langpack.create, "Language pack creation function not present")
end

function suite.tryfallback()
   local expn = tongue.langpack.fallback:expand("TEST", {foo="bar"})
   assert(string.sub(expn, 1, 10) == "!!!TEST!!!", expn)
   assert(string.find(expn, 'foo="bar"'), expn)
end

function suite.tryparentfallback()
   local lang = tongue.langpack.create("en")
   assert(lang, "Could not create language pack")
   local expn = lang:expand("TEST", {foo="bar"})
   assert(string.sub(expn, 1, 10) == "!!!TEST!!!", expn)
   assert(string.find(expn, 'foo="bar"'), expn)
end

function suite.add_function_expn()
   local lang = tongue.langpack.create("en")
   assert(lang, "Could not create language pack")
   assert(lang:add_token("TEST", function(args) return args.foo end),
	  "Unable to add a function token")
   assert(lang:expand("TEST",{foo="bar"}) == "bar", "Function tokens don't work")
end

function suite.add_plain_expn_good_lazy()
   local lang = tongue.langpack.create("en")
   assert(lang, "Could not create language pack")
   assert(lang:add_token("TEST", "Hello World"),
	  "Unable to add a good plain token")
   assert(lang:expand("TEST",{}) == "Hello World", "Good plain text tokens don't work")
end

function suite.add_plain_expn_good_strict()
   local lang = tongue.langpack.create("en")
   assert(lang, "Could not create language pack")
   assert(lang:add_token("TEST", "Hello World", true),
	  "Unable to add a good plain token strictly")
   assert(lang:expand("TEST",{}) == "Hello World", "Good plain text tokens don't work")
end

function suite.add_simple_expn_good_lazy()
   local lang = tongue.langpack.create("en")
   assert(lang, "Could not create language pack")
   assert(lang:add_token("TEST", "Hello World"),
	  "Unable to add a good plain token")
   assert(lang:add_token("TEST2", "I say $(TEST)"), "Unable to add a good simple token")
   assert(lang:expand("TEST2",{}) == "I say Hello World", "Good simple text tokens don't work")
end

function suite.add_simple_expn_good_strict()
   local lang = tongue.langpack.create("en")
   assert(lang, "Could not create language pack")
   assert(lang:add_token("TEST", "Hello World", true),
	  "Unable to add a good plain token")
   assert(lang:add_token("TEST2", "I say $(TEST)", true), "Unable to add a good simple token")
   assert(lang:expand("TEST2",{}) == "I say Hello World", "Good simple text tokens don't work")
end

function suite.add_simple_expn_good_pass()
   local lang = tongue.langpack.create("en")
   assert(lang, "Could not create language pack")
   assert(lang:add_token("TEST", "Hello ${name}", true),
	  "Unable to add a good plain token")
   assert(lang:add_token("TEST2", "I say $(TEST,...)", true), "Unable to add a good simple token")
   assert(lang:expand("TEST2",{name="Jeff"}) == "I say Hello Jeff", "Good simple text tokens don't work")
end

function suite.add_simple_expn_good_pass_override()
   local lang = tongue.langpack.create("en")
   assert(lang, "Could not create language pack")
   assert(lang:add_token("TEST", "Hello ${name}", true),
	  "Unable to add a good plain token")
   assert(lang:add_token("TEST2", "I say $(TEST,...,name=Susan)", true), "Unable to add a good simple token")
   assert(lang:expand("TEST2",{name="Jeff"}) == "I say Hello Susan", "Good simple text tokens don't work")
end

function suite.add_simple_expn_good_pass_onlyone()
   local lang = tongue.langpack.create("en")
   assert(lang, "Could not create language pack")
   assert(lang:add_token("TEST", "Hello ${title} ${name}", true),
	  "Unable to add a good plain token")
   assert(lang:add_token("TEST2", "I say $(TEST,name)", true), "Unable to add a good simple token")
   assert(lang:expand("TEST2",{name="Jeff", title="Mr"}) == "I say Hello  Jeff", "Good simple text tokens don't work")
end

function suite.add_simple_expn_good_pass_puredata()
   local lang = tongue.langpack.create("en")
   assert(lang, "Could not create language pack")
   assert(lang:add_token("TEST", "Hello ${name}", true),
	  "Unable to add a good plain token")
   assert(lang:add_token("TEST2", "I say $(TEST,name=Maria)", true), "Unable to add a good simple token")
   assert(lang:expand("TEST2",{}) == "I say Hello Maria", "Good simple text tokens don't work")
end


local count_ok = 0
for _, testname in ipairs(testnames) do
--   print("Run: " .. testname)
   local ok, err = xpcall(suite[testname], debug.traceback)
   if not ok then
      print(err)
      print()
   else
      count_ok = count_ok + 1
   end
end

print(tostring(count_ok) .. "/" .. tostring(#testnames) .. " [" .. tostring(total_asserts) .. "] OK")

os.exit(count_ok == #testnames and 0 or 1)
