typedload
=========

Load and dump json-like data into typed data structures in Python3, enforcing
a schema on the data.

This module provides an API to load dictionaries and lists (usually loaded
from json) into Python's NamedTuples, dataclass, sets, enums, and various
other typed data structures; respecting all the type-hints and performing
type checks or casts when needed.

It can also dump from typed data structures to json-like dictionaries and lists.

It is very useful for projects that use Mypy and deal with untyped data
like json, because it guarantees that the data will follow the specified schema.

Note that it is released with a GPL license and it cannot be used inside non
GPL software.

Example
=======

For example this dictionary, loaded from a json:

```python
data = {
    'users': [
        {
            'username': 'salvo',
            'shell': 'bash',
            'sessions': ['pts/4', 'tty7', 'pts/6']
        },
        {
            'username': 'lop'
        }
    ],
}
```


Can be treated more easily if loaded into this type:

```python
class User(NamedTuple):
    username: str
    shell: str = 'bash'
    sessions: List[str] = []

class Logins(NamedTuple):
    users: List[User]
```

And the data can be loaded into the structure with this:

```python
t_data = typedload.load(data, Logins)
```

And then converted back:

```python
data = typedload.dump(t_data)
```

Supported types
===============

Since this is not magic, not all types are supported.

The following things are supported:

 * Basic python types (int, str, bool, float, NoneType)
 * NamedTuple
 * Enum
 * Optional[SomeType]
 * List[SomeType]
 * Dict[TypeA, TypeB]
 * Tuple[TypeA, TypeB, TypeC]
 * Set[SomeType]
 * Union[TypeA, TypeB]
 * dataclass (requires Python 3.7)
 * attr.s
 * ForwardRef (Refer to the type in its own definition)
 * Literal (requires Python 3.8)
 * TypedDict (requires Python 3.8)
 * datetime.date, datetime.time, datetime.datetime

Advantage when using Mypy
=========================

```python
# This is treated as Any, no checks done.
data = json.load(f)

# This is treated as Dict[str, int]
# but there will be runtime errors if the data does not
# match the expected format
data = json.load(f)  # type: Dict[str, int]

# This is treated as Dict[str, int] and an exception is
# raised if the actual data is not Dict[str, int]
data = typedload.load(json.load(f), Dict[str, int])
```

So when using Mypy, it makes sense to make sure that the type is correct,
rather than hoping the data will respect the format.

Documentation
=============

The documentation can be generated by running:

```
make docs
```

And it will be located inside the `docs` directory.

See the file `example.py` to see a basic usecase for this module.

The tests are harder to read but provide more in depth examples of
the capabilities of this module.
